///
/// @file
/// @details Provides some error definitions, functions and handlers.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_error.h"
#include "tb_types.h"
#include "implementation/tbi_diagnostics.h"

#include <stdexcept>
#include <string>
#include <list>
#include <cstdarg>
#include <cstdio>
#include <stdio.h>
#include <iostream>
#include <cassert>

namespace tbImplementation
{

	///
	///	@details This error handler should be used in situations where an error is expected to be worked around by
	///	trying to catch an exception and handle it.  Very useful for application development, still useful but less
	///	used within game development.
	///
	///	Hidden from public API as it is expected to be enabled/disabled through the Error:: namespace API.
	///
	class ExceptionalErrorHandler : public tbCore::Error::ErrorHandlerInterface
	{
	public:
		class ErroredException : public std::runtime_error
		{
		public:
			explicit ErroredException(const std::string& whatHappened) :
				std::runtime_error(whatHappened.c_str())
			{
			}

		private:
		};

		virtual ~ExceptionalErrorHandler(void) { }
		virtual void OnErrorFired(const std::string& errorMessage)
		{
			ErroredException exception(errorMessage);
			throw exception;
		}
	};

	void OnTriggerError(const char* const fromFile, int onLineNumber, const char* const formattedString, va_list variableArguments);
	void OnErrorFired(const std::string& fromFile, int onLineNumber, const std::string& errorMessage);

	typedef std::list<tbCore::Error::ErrorHandlerInterface*> ErrorHandlerContainer;
	ErrorHandlerContainer tbiErrorHandlers;
	ExceptionalErrorHandler tbiExceptionHandler;
	bool tbiUseExceptionalHandler(false);
	bool tbiUseErrorHandler(true);
	int tbiErrorFiredOnLineNumber(-1);
	std::string tbiErrorFiredInFileNamed("");

};	/* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OnErrorFired(const std::string& fromFile, int onLineNumber, const std::string& errorMessage)
{
	tbiErrorFiredOnLineNumber = onLineNumber;
	tbiErrorFiredInFileNamed = fromFile;

	if (true == tbiUseExceptionalHandler)
	{
		tbiExceptionHandler.OnErrorFired(errorMessage);
	}

	//std::cout << "\nASSERTION FAILED! -----------------------------------\n";
	//std::cout << "FILE " << fromFile << "[" << onLineNumber << "]\n";
	//std::cout << "MESSAGE: " << errorMessage << "\n";
	//std::cout << "ASSERTION FAILED! -----------------------------------\n";
	//std::cout.flush();

	tbi_log_error("\n\n----- Error Condition Triggered -----\n");
	tbi_log_error("FILE: %s[%d]\n", fromFile.c_str(), onLineNumber);
	tbi_log_error("MESSAGE: %s\n", errorMessage.c_str());
	tbi_log_error("----- Error Condition Triggered -----\n");
	tbi_log_flush();

	for (ErrorHandlerContainer::iterator itr = tbiErrorHandlers.begin(), itrEnd = tbiErrorHandlers.end(); itr != itrEnd; ++itr)
	{
		(*itr)->OnErrorFired(errorMessage);
	}

	if (true == tbiUseErrorHandler)
	{
		assert(0 && errorMessage.c_str());
	}
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbCore::Error::ErrorHandlerInterface::ErrorHandlerInterface(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::Error::ErrorHandlerInterface::~ErrorHandlerInterface(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

int tbCore::Error::ErrorHandlerInterface::GetLineNumber(void) const
{
	return tbImplementation::tbiErrorFiredOnLineNumber;
}

//--------------------------------------------------------------------------------------------------------------------//

const std::string& tbCore::Error::ErrorHandlerInterface::GetFileName(void) const
{
	return tbImplementation::tbiErrorFiredInFileNamed;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::AddErrorHandler(tbCore::Error::ErrorHandlerInterface* errorHandler)
{
	tbImplementation::tbiErrorHandlers.push_back(errorHandler);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::RemoveErrorHandler(tbCore::Error::ErrorHandlerInterface* errorHandler)
{
	tbImplementation::tbiErrorHandlers.remove(errorHandler);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::EnableThrowOnError(void)
{
	tbImplementation::tbiUseExceptionalHandler = true;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::DisableThrowOnError(void)
{
	tbImplementation::tbiUseExceptionalHandler = false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OnTriggerError(const char* const fromFile, int onLineNumber, const char* const formattedString, va_list variableArguments)
{
	const size_t kErrorBufferSize(4096);
	char buffer[kErrorBufferSize] = { '\0', };

#if defined(tb_windows)	//Because Microsoft likes being difficult.
  #if defined(tb_visual_cpp)
	vsprintf_s(buffer, kErrorBufferSize, formattedString, variableArguments);
  #else
	vsprintf(buffer, formattedString, variableArguments);
  #endif
#else
	vsprintf(buffer, formattedString, variableArguments);
#endif
	va_end(variableArguments);

	const std::string errorMessage(buffer);

	//The real magic for an error being triggered happens here.
	tbImplementation::OnErrorFired(fromFile, onLineNumber, errorMessage);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::TriggerError(const char* const formattedString, ...)
{
	va_list argumentsList;
	va_start(argumentsList, formattedString);
	tbImplementation::OnTriggerError("unknown file", 0, formattedString, argumentsList);
	va_end(argumentsList);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::TriggerError(const char* const fromFile, int onLineNumber, const char* const formattedString, ...)
{
	va_list argumentsList;
	va_start(argumentsList, formattedString);
	tbImplementation::OnTriggerError(fromFile, onLineNumber, formattedString, argumentsList);
	va_end(argumentsList);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::TriggerErrorIf(bool testResult, const char* const formattedString, ...)
{
	if (true == testResult)
	{
		va_list argumentsList;
		va_start(argumentsList, formattedString);
		tbImplementation::OnTriggerError("unknown_file", 0, formattedString, argumentsList);
		va_end(argumentsList);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::Error::TriggerErrorIf(bool testResult, const char* const fromFile, int onLineNumber, const char* const formattedString, ...)
{
	if (true == testResult)
	{
		va_list argumentsList;
		va_start(argumentsList, formattedString);
		tbImplementation::OnTriggerError(fromFile, onLineNumber, formattedString, argumentsList);
		va_end(argumentsList);
	}
}

//--------------------------------------------------------------------------------------------------------------------//
