///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_String_h_
#define _TurtleBrains_String_h_

#include "tb_configuration.h"
#include "tb_error.h"

#include <sstream>
#include <string>

///
/// @defgroup String String Conversion Macros
/// @{
///

///
/// @details An easy way to make sure that a string is processed correctly is to just call tb_string on all strings
///   passed to TurtleBrains so that both ascii and wide string arrays will convert into the correct std::string or
///   std::wstring type depending on the the tb_wide_string setting in tb_configuration.h
///
#define tb_string(object)  TurtleBrains::Core::ToString((object))

///
///	@}
///

namespace TurtleBrains
{
	namespace Core
	{

		///
		/// @details Convert some type of an object into a std::string using the operator<<(ostream) which must exist in
		///   for the type specified by T.
		///
		/// @param object    The object to convert into a string, typically a basic-type, (int, float, bool), but any type
		///   with a defined operator<<(ostream) will work.
		///
		/// @return the object transformed into a std::string.
		///
		/// @note If the object type, T, does not have an operator<<(ostream) the compiler will likely complain.
		///
		template <typename T> std::string ToStdString(const T& object)
		{
			std::stringstream ss;
			ss << object;
			return ss.str();
		}

		///
		/// @details This is a specialization of the template function above, if the type specified by T is already a
		///   std::string, simply return a value of that string without needing conversion.
		///
		/// @param object    The string to return
		///
		/// @return a value of the string represented by object.
		///
		inline std::string ToStdString(const std::string& object)
		{
			return object;
		}

		///
		/// @details This is a specialization of the template function ToStdString, if the type specified by T is a
		///   wide-string (std::wstring), attempt to conver the characters into a std::string the best way possible to
		///   retain as much of the initial string as possible, replacing unknown or unsupported characters with a
		///   question mark '?' character.
		///
		/// @param object    The wide-string to covert into a std::string
		///
		inline std::string ToStdString(const std::wstring& object)
		{
			std::stringstream stream;
			const std::ctype<char>& ctfacet = std::use_facet< std::ctype<char> >(stream.getloc());
			for (size_t i = 0; i < object.size(); ++i)
			{
				stream << static_cast<char>(ctfacet.narrow(static_cast<char>(object[i]), '?'));
			}
			return stream.str();
		}

		///
		/// @details This is a specialization of the template function ToStdString, if the type specified by T is a c-style
		///   string of wchar's, attempt to convert the characters into a std::string by first converting it into a
		///   wide-string and reusing the ToStdString(const std::wstring&) code.
		///
		/// @param object    The c-style wide-string to covert into a std::string.
		///
		inline std::string ToStdString(const wchar_t* object)
		{
			return ToStdString(std::wstring(object));
		}

		///
		/// @details This is a specialization of the template function ToWideString to provide specific precision values when
		///   dealing with floating point numbers, so ToWideString(value, 3) would print out: value to the third decimal point
		///   like: 52.123
		///
		/// @param value The floating point value that will be converted into a std::string.
		/// @param precision How many points AFTER the decimal point should be displayed, defaults to -1 which means use the
		///   defaults of stringstream.
		///
		inline std::string ToStdString(float value, int precision = -1)
		{
			std::stringstream ss;
			if (precision >= 0)
			{
				ss.precision(precision);
				ss << std::fixed;
			}
			ss << value;
			return ss.str();
		}

		///
		/// @details Convert a std::string into some type of an object, specified by type T, using the operator>>(istream)
		///   which must exist in for the type specified by T as must a default constructor.  For best portability and
		///   reducing non-wide to wide string conversions, using the function FromString() should be used instead.
		///
		/// @param input     The string that will be used to create an object.
		///
		/// @return the object of type T with the value setup/converted from a std::string using the operator>>(istream).
		///
		/// @note If the object type, T, does not have an operator>>(istream) or default constructor the compiler will
		///   almost certainly complain.
		///
		template <typename T> T FromStdString(const std::string& input)
		{
			tb_error_if(true == input.empty(), "Invalid value for parameter: input, expected valid, non-empty, string.");
			T object;
			std::stringstream ss(input);
			ss >> object;
			return object;
		}

		///
		/// @details Convert a std::string into some type of an object, specified by type T, using the operator>>(istream)
		///   which must exist in for the type specified by T as must a default constructor.
		///
		/// @param input     The string that will be used to create an object.
		///
		/// @return the object of type T with the value setup/converted from a std::string using the operator>>(istream).
		///
		/// @note If the object type, T, does not have an operator>>(istream) or default constructor the compiler will
		///   almost certainly complain.
		///
		template <typename T> T FromString(const std::string& input)
		{
			return FromStdString<T>(input);
		}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details Convert some type of an object into a std::wstring using the operator<<(wostream) which must exist in
		///   for the type specified by T.
		///
		/// @param object    The object to convert into a wide-string, typically a basic-type, (int, float, bool), but any
		///   type with a defined operator<<(wostream) will work.
		///
		/// @return the object transformed into a std::wstring.
		///
		/// @note If the object type, T, does not have an operator<<(wostream) the compiler will likely complain.
		///
		template <typename T> std::wstring ToWideString(const T& object)
		{
			std::wstringstream wss;
			wss << object;
			return wss.str();
		}

		///
		/// @details This is a specialization of the template function ToWideString, if the type specified by T is already
		///   a std::wstring, simply return a value of that string without needing conversion.
		///
		/// @param object    The wide-string to return
		///
		/// @return a value of the wide-string represented by object.
		///
		inline std::wstring ToWideString(const std::wstring& object)
		{
			return object;
		}

		///
		/// @details This is a specialization of the template function ToWideString, if the type specified by T is a
		///   std::string, attempt to convert the characters into a std::wstring the best way possible by widening each
		///   character to retain the same initial string in widened format.
		///
		/// @param object    The string to covert into a wide string, (std::wstring)
		///
		inline std::wstring ToWideString(const std::string& object)
		{
			std::wostringstream stream;
			const std::ctype<wchar_t>& ctfacet = std::use_facet< std::ctype<wchar_t> >(stream.getloc());
			for (size_t i = 0; i < object.size(); ++i)
			{
				stream << ctfacet.widen(object[i]);
			}
			return stream.str();
		}

		///
		/// @details This is a specialization of the template function ToWideString, if the type specified by T is a
		///   c-style string of char's, attempt to convert the characters into a std::wstring by first converting it into a
		///   std::string and reusing the ToWideString(const std::string&) code.
		///
		/// @param object    The c-style string to convert into a std::string.
		///
		inline std::wstring ToWideString(const char* object)
		{
			return ToWideString(std::string(object));
		}

		///
		/// @details This is a specialization of the template function ToWideString to provide specific precision values when
		///   dealing with floating point numbers, so ToWideString(value, 3) would print out: value to the third decimal point
		///   like: 52.123
		///
		/// @param value The floating point value that will be converted into a wide string.
		/// @param precision How many points AFTER the decimal point should be displayed, defaults to -1 which means use the
		///   defaults of stringstream.
		///
		inline std::wstring ToWideString(float value, int precision = -1)
		{
			std::wstringstream wss;
			if (precision >= 0)
			{
				wss.precision(precision);
				wss << std::fixed;
			}
			wss << value;
			return wss.str();
		}

		///
		/// @details Convert a std::wstring into some type of an object, specified by type T, using operator>>(wistream)
		///   which must exist in for the type specified by T as must a default constructor.  For best portability and
		///   reducing non-wide to wide string conversions, using the function FromString() should be used instead.
		///
		/// @param input     The wide-string that will be used to create an object of type T
		///
		/// @return the object of type T with the value setup/converted from a std::wstring using the operator>>(wistream).
		///
		/// @note If the object type, T, does not have an operator>>(wistream) or default constructor the compiler will
		///   almost certainly complain.
		///
		template <typename T> T FromWideString(const std::wstring& input)
		{
			tb_error_if(true == input.empty(), "Invalid value for parameter: input, expected valid, non-empty, string.");
			T object;
			std::wstringstream wss(input);
			wss >> object;
			return object;
		}

		///
		/// @details Convert a std::wstring into some type of an object, specified by type T, using operator>>(wistream)
		///   which must exist in for the type specified by T as must a default constructor.
		///
		/// @param input     The wide-string that will be used to create an object of type T
		///
		/// @return the object of type T with the value setup/converted from a std::wstring using the operator>>(wistream).
		///
		/// @note If the object type, T, does not have an operator>>(wistream) or default constructor the compiler will
		///   almost certainly complain.
		///
		template <typename T> T FromString(const std::wstring& input)
		{
			return FromWideString<T>(input);
		}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details The tbString is a type definition of either wide-string (std::wstring) when tb_wide_string is defined
		///   or std::string when tb_wide_string is not defined.  This effects how TurtleBrains strings are held and passed
		///   The tb_wide_string can be defined in tb_configuration.h if wide-strings are preferred.
		///
		/// @note For maximum portability, when using strings returned from TurtleBrains you should not assume the tbString
		///   is defined as STD or WIDE and instead always convert it back to what you are expecting.  While this may add a
		///   small performance overhead, it will allow you to sleep at night knowing if you decide to change from STD to
		///   WIDE strings, everything will work splendidly.  Alternatively, storing all your own strings as tbString will
		///   also allow for the same peace of mind.
		///
#ifdef tb_wide_string
		typedef std::wstring tbString;
//		class tbString : public std::wstring
//		{
//		public:
//			tbString(void) : std::wstring(L"") { }
//			tbString(const tbString& input) : std::wstring(input.c_str()) { }
//			tbString(const std::wstring& input) : std::wstring(input) { }
//			tbString(const std::string& input) : std::wstring(ToWideString(input)) { }
//			tbString(const wchar_t* input) : std::wstring(input) { }
//			tbString(const char* input) : std::wstring(ToWideString(input)) { }
//
//			tbString& operator=(const tbString& input) { std::wstring::operator=(input.c_str()); return *this; }
//			tbString& operator=(const std::wstring& input) { std::wstring::operator=(input); return *this; }
//			tbString& operator=(const std::string& input) { std::wstring::operator=(ToWideString(input)); return *this; }
//			tbString& operator=(const wchar_t* input) { std::wstring::operator=(input); return *this; }
//			tbString& operator=(const char* input) { std::wstring::operator=(ToWideString(input)); return *this; }
//
//			friend tbString operator+(const std::string& leftSide, const tbString& rightSide) { return ToWideString(leftSide) + rightSide.c_str(); }
//			friend tbString operator+(const char* leftSide, const tbString& rightSide) { return ToWideString(leftSide) + rightSide.c_str(); }
//
//		//			tbString operator+(const tbString& input) { return std::wstring::operator+(std::wstring(input.c_str())); }
//		//			tbString operator+(const std::wstring& input) { return std::wstring::operator+(input); }
//		//			tbString operator+(const std::string& input) { return std::wstring::operator+(ToWideString(input)); }
//		//			tbString operator+(const wchar_t* input) { return std::wstring::operator+(input); }
//		//			tbString operator+(const char* input) { return std::wstring::operator+(ToWideString(input.c_str())); }
//
//			operator const void*(void) const { return c_str(); }
//
//			//operator const wchar_t*(void) const { return c_str(); }
//
//			//std::istream& operator>>(std::istream& inputStream) { std::wstring s; inputStream >> s; *this = s; return inputStream; }
////			std::wistream& operator>>(std::wistream& inputStream) { std::wstring s; inputStream >> s; *this = s; return inputStream; }
////			std::wostream& operator<<(std::wostream& outputStream) { outputStream << this->c_str(); return outputStream; } // s; *this = s; return inputStream; }
////			std::istream& operator>>(std::istream& inputStream) { std::string s; inputStream >> s; *this = s; return inputStream; }
////			std::ostream& operator<<(std::ostream& outputStream) { std::string s = ToStdString(this->c_str()); outputStream << s; return outputStream; } // s; *this = s; return inputStream; }
//		};
#else
		typedef std::string tbString;
//	class tbString : public std::string
//	{
//	public:
//		tbString(void) : std::string("") { }
//		tbString(const tbString& input) : std::string(input.c_str()) { }
//		tbString(const std::wstring& input) : std::string(ToStdString(input)) { }
//		tbString(const std::string& input) : std::string(input) { }
//		tbString(const wchar_t* input) : std::string(ToStdString(input)) { }
//		tbString(const char* input) : std::string(input) { }
//
//		tbString& operator=(const tbString& input) { std::string::operator=(input.c_str()); return *this; }
//		tbString& operator=(const std::wstring& input) { std::string::operator=(ToStdString(input)); return *this; }
//		tbString& operator=(const std::string& input) { std::string::operator=(input); return *this; }
//		tbString& operator=(const wchar_t* input) { std::string::operator=(ToStdString(input)); return *this; }
//		tbString& operator=(const char* input) { std::string::operator=(input); return *this; }
//
//		friend tbString operator+(const std::wstring& leftSide, const tbString& rightSide) { return ToStdString(leftSide) + rightSide.c_str(); }
//		friend tbString operator+(const wchar_t* leftSide, const tbString& rightSide) { return ToStdString(leftSide) + rightSide.c_str(); }
//
//		operator const void*(void) const { return c_str(); }
//	};

#endif /* tb_wide_string */

		///
		/// @details Easily convert to the string type defined by tbString, which is either std::string or std::wstring
		///   depending on the tb_wide_string definition.  By using this to convert when passing string parameters into
		///   the TurtleBrains framework you can sleep well knowing it will be converted and held safely regardless of
		///   the tb_wide_string definition and the string type passed.  Even objects with both operator<<(ostream) and
		///   operator<<(wostream) can be converted into a tbString using ToString().
		///
		/// @note Non-string objects like basic primitives and even custom types can be converted to a tbString using
		///   ToString but will need to have both operator<<(ostream) and operator<<(wostream) defined and implemented to
		///   perform the conversion.
		///
		template <typename T> tbString ToString(const T& object)
		{
#ifdef tb_wide_string
			return ToWideString(object);
#else
			return ToStdString(object);
#endif /* tb_wide_string */
		}

		inline tbString ToString(float value, int precision = 0)
		{
#ifdef tb_wide_string
			return ToWideString(value, precision);
#else
			return ToStdString(value, precision);
#endif /* tb_wide_string */
		}

	}; /* namespace Core */
}; /* namespace TurtleBrains */

//			inline operator float*(void) { return mFloatArray; }

//std::string& operator=(std::string& leftSide, const std::wstring& rightSide) { lhs = TurtleBrains::Core::ToStdString(rightSide); return leftSide; }
//std::wstring& operator=(std::wstring& leftSide, const std::string& rightSide) { lhs = TurtleBrains::Core::ToWideString(rightSide); return rightSide; }

//operator() std::string(const std::wstring& input) { return TurtleBrains::Core::ToStdString(input); }
//std::wstring operator(const std::string& input) { return TurtleBrains::Core::ToWideString(input); }

namespace tbCore = TurtleBrains::Core;

#endif /* _TurtleBrains_String_h_ */
