///
/// @file
/// @details Create a console window where stdout will be redirected for printf, cout and cerr to be viewed in real-time
///   as the project is being run.  Output will also be dumped into a lot for post-run debugging.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_debug_logger.h"

#include <iostream>
#include <fstream>

#if defined(tb_windows)
  #define WIN32_LEAN_AND_MEAN
  #include <windows.h>
  #include <io.h>
  #include <fcntl.h>
  #include <cstdio>
#endif /* tb_windows */

#if defined(tb_linux)
  #include <cstdarg>
#endif /* tb_linux */

enum DisplayAttribute
{
	kDisplayText = 7,
	kDisplayImportant = 15,
	kDisplayTest = 9,
	kDisplayPass = 10,
	kDisplayFail = 12,
	kDisplayWarn = 6, //2|4
};

namespace tbImplementation
{
	void SetupDebugConsole(short bufferWidth, short bufferHeight, short windowWidth, short windowHeight);
	void SetConsoleAttr(const DisplayAttribute& display);
	void DoLogging(const char *szFormat, va_list va);
	void CreateConsoleLogger(void);

	std::string tbiLogPath;
	std::ofstream tbiLogOut;
};

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetConsoleAttr(const DisplayAttribute& attribute)
{
#if defined(tb_windows)
	SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE), attribute);
#endif /* tb_windows */
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::DoLogging(const char* formattedString, va_list variableArguments)
{
	const size_t kLogBufferSize(8192);
	char buffer[kLogBufferSize] = { '\0', };

#if defined(tb_windows)
  #if defined(tb_visual_cpp)
	vsprintf_s(buffer, kLogBufferSize, formattedString, variableArguments);
  #else
	vsprintf(buffer, formattedString, variableArguments);
  #endif

	//freopen("CON", "w", stdout);
	//freopen("CON", "w", stderr);
#else
	vsprintf(buffer, formattedString, variableArguments);
#endif

	std::string logMessage(buffer);

	printf("%s", logMessage.c_str());
	std::cout << logMessage << std::flush;
	//std::cerr << logMessage << std::flush;

#if defined(tb_windows) && defined(tb_visual_cpp)
	OutputDebugString(logMessage.c_str());
#endif /* tb_windows */

	if (true == tbImplementation::tbiLogOut.is_open())
	{
		tbImplementation::tbiLogOut << logMessage << std::flush;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CreateConsoleLogger(void)
{
	SetupDebugConsole(80, 9999, 80, 75);
//	SetConsoleAttr(kDisplayText);
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Debug::OpenLog(const std::string& logFile, bool createConsoleWindow)
{
	if (true == createConsoleWindow)
	{
		tbImplementation::CreateConsoleLogger();
	}

	if (true == tbImplementation::tbiLogOut.is_open())
	{
		return;
	}

	tbImplementation::tbiLogPath = logFile;
	tbImplementation::tbiLogOut.open(tbImplementation::tbiLogPath.c_str(), std::ios::trunc | std::ios::out);
	if (false == tbImplementation::tbiLogOut.is_open())
	{	//Although the log file didn't open, they may have created a console window and see this message.
		Log("Failed to open file for log: \"%s\".\n", tbImplementation::tbiLogPath.c_str());
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Debug::SaveLog(void)
{
	if (true == tbImplementation::tbiLogOut.is_open())
	{
		tbImplementation::tbiLogOut.flush();
		tbImplementation::tbiLogOut.close();
		tbImplementation::tbiLogOut.open(tbImplementation::tbiLogPath.c_str(), std::ios::app | std::ios::out);

		if (false == tbImplementation::tbiLogOut.is_open())
		{	//Although the log file didn't open, they may have created a console window and see this message.
			Log("Failed to open Log: \"%s\" after saving.\n", tbImplementation::tbiLogPath.c_str());
		}
	}

	std::cout.flush();
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Debug::CloseLog(void)
{
	if (true == tbImplementation::tbiLogOut.is_open())
	{
		tbImplementation::tbiLogOut.flush();
		tbImplementation::tbiLogOut.close();
	}

	//TODO: TIM: Find a way to close the console window as well?
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Debug::Log(const char* formattedString, ...)
{
	va_list argumentsList;
	va_start(argumentsList, formattedString);
	tbImplementation::DoLogging(formattedString, argumentsList);
	va_end(argumentsList);
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Debug::LogIf(const bool testResult, const char* formattedString, ...)
{
	if (true == testResult)
	{
		va_list argumentsList;
		va_start(argumentsList, formattedString);
		tbImplementation::DoLogging(formattedString, argumentsList);
		va_end(argumentsList);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogText(const char *szFormat, ...)
//{
//	va_list va;
//	va_start(va, szFormat);
//	tbImplementation::DoLogging(szFormat, va);
//	va_end(va);
//}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogIText(const char *szFormat, ...)
//{
//	va_list variableArgumentList;
//	va_start(variableArgumentList, szFormat);
//	tbImplementation::SetConsoleAttr(kDisplayImportant);
//	tbImplementation::DoLogging(szFormat, variableArgumentList);
//	tbImplementation::SetConsoleAttr(kDisplayText);
//	va_end(variableArgumentList);
//}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogTest(const char *szFormat, ...)
//{
//	va_list variableArgumentList;
//	va_start(variableArgumentList, szFormat);
//	tbImplementation::SetConsoleAttr(kDisplayTest);
//	tbImplementation::DoLogging(szFormat, variableArgumentList);
//	tbImplementation::SetConsoleAttr(kDisplayText);
//	va_end(variableArgumentList);
//}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogPass(const char *szFormat, ...)
//{
//	va_list variableArgumentList;
//	va_start(variableArgumentList, szFormat);
//	tbImplementation::SetConsoleAttr(kDisplayPass);
//	tbImplementation::DoLogging(szFormat, variableArgumentList);
//	tbImplementation::SetConsoleAttr(kDisplayText);
//	va_end(variableArgumentList);
//}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogFail(const char *szFormat, ...)
//{
//	va_list variableArgumentList;
//	va_start(variableArgumentList, szFormat);
//	tbImplementation::SetConsoleAttr(kDisplayFail);
//	tbImplementation::DoLogging(szFormat, variableArgumentList);
//	tbImplementation::SetConsoleAttr(kDisplayText);
//	va_end(variableArgumentList);
//}

//--------------------------------------------------------------------------------------------------------------------//

//void TurtleBrains::Debug::LogWarn(const char *szFormat, ...)
//{
//	va_list variableArgumentList;
//	va_start(variableArgumentList, szFormat);
//	tbImplementation::SetConsoleAttr(kDisplayWarn);
//	tbImplementation::DoLogging(szFormat, variableArgumentList);
//	tbImplementation::SetConsoleAttr(kDisplayText);
//	va_end(variableArgumentList);
//}

//--------------------------------------------------------------------------------------------------------------------//

//Taken from DirectX class from Full Sail Real World Education...
//=============================================================================
//=============================================================================
// DebugConsole.cpp
//
// Utility to popup & utilize console window in GUI application.
//
// taken from:
// http://dslweb.nwnexus.com/~ast/dload/guicon.htm
//=============================================================================
//=============================================================================

//Original Includes moved to the top of the .cpp by Tim Beaudet

///////////////////////////////////////////////////////////////////////////////
// SetupDebugConsole
//
// Function to allocate and display conosle window & to re-route all standard
// input, ouput & error streams to it.
///////////////////////////////////////////////////////////////////////////////

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetupDebugConsole(short bufferWidth, short bufferHeight, short windowWidth, short windowHeight)
{
#if defined(tb_windows)
#if defined(tb_visual_cpp) //This should technically work on Windows.  _fdopen undeclared errors though.

	// locals
	CONSOLE_SCREEN_BUFFER_INFO  coninfo;
	//FILE                       *pFile;
//	int                         conHandle;
//	HANDLE                      stdHandle;
	SMALL_RECT                  window = {0,};

	// allocate console
	AllocConsole();

	// reset console properties
	GetConsoleScreenBufferInfo( GetStdHandle( STD_OUTPUT_HANDLE ), &coninfo );
	coninfo.dwSize.Y = 1024;
	coninfo.dwSize.X = bufferWidth;
	window.Left      = 0;
	window.Top       = 0;
	window.Right     = windowWidth-1;
	window.Bottom    = windowHeight-1;
	SetConsoleScreenBufferSize( GetStdHandle( STD_OUTPUT_HANDLE ), coninfo.dwSize );
	SetConsoleWindowInfo( GetStdHandle( STD_OUTPUT_HANDLE ), true, &window );

	// redirect STDOUT to console
	//stdHandle = GetStdHandle( STD_OUTPUT_HANDLE );
	//conHandle = _open_osfhandle( (intptr_t)stdHandle, _O_TEXT );
	//pFile = _fdopen( conHandle, "w" );
	//*stdout = *pFile;
	//setvbuf( stdout, NULL, _IONBF, 0 ); // unbuffered

	//// redirect STDIN to console
	//stdHandle = GetStdHandle( STD_INPUT_HANDLE );
	//conHandle = _open_osfhandle( (intptr_t)stdHandle, _O_TEXT );
	//pFile = _fdopen( conHandle, "r" );
	//*stdin = *pFile;
	//setvbuf( stdin, NULL, _IONBF, 0 ); // unbuffered

	//// redirect STDERR to console
	//stdHandle = GetStdHandle( STD_ERROR_HANDLE );
	//conHandle = _open_osfhandle( (intptr_t)stdHandle, _O_TEXT );
	//pFile = _fdopen( conHandle, "w" );
	//*stderr = *pFile;
	//setvbuf( stderr, NULL, _IONBF, 0 ); // unbuffered

	// route cout, wcout, cin, wcin, wcerr, cerr, wclog & clog as well
	std::ios::sync_with_stdio();

	freopen("CON", "r", stdin);
	freopen("CON", "w", stdout);
	freopen("CON", "w", stderr);

#endif /* tb_visual_cpp */
#endif /* tb_windows */
}

//--------------------------------------------------------------------------------------------------------------------//
