///
/// @file
/// @details Create a console window where stdout will be redirected for printf, cout and cerr to be viewed in real-time
///   as the project is being run.  Output will also be dumped into a lot for post-run debugging.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_DebugLogger_h_
#define _TurtleBrains_DebugLogger_h_

#include "tb_debug_tool_set.h"
#include <string>

///
/// @defgroup Debug Debug Log Macros
/// @{
///

#ifdef tb_use_debug_set

	///
	///	@details Opens a file to output the log at, and can, on Windows building with Visual CPP, open a console
	///   window for showing std out if the console parameter is passed as true.  Underneath this simply calls
	///	  the Debug::OpenLog() function, however the define should be prefferred when you want to have the option of
	///   turning off all logging with the tb_disable_debug_set define in tb_configuration.h
	///
	/// @param logFile The filepath of the location to save the log file.  This file will be cleared when loaded so the
	///   contents will only show a single run.
	/// @param createConsoleWindow Set to true this will attempt to create a console window.  Currently only supported on Windows
	///   when building with Visual CPP.
	///
	#define tb_log_open(logFile, createConsoleWindow) TurtleBrains::Debug::OpenLog(logFile, createConsoleWindow)

	///
	///	@details Saves the log file to disk to make sure any log messages have been flushed.  Underneath this simply 
	///   calls the Debug::SaveLog() function, however the define should be prefferred when you want to have the option
	///   of turning off all logging with the tb_disable_debug_set define in tb_configuration.h
	///
	#define tb_log_save() TurtleBrains::Debug::SaveLog()

	///
	///	@details Closes the log file after flushing any pending log messages, once closed any new log messages will not
	///   be written to the log file, only to the stdout.  Underneath this simply calls the Debug::CloseLog() function,
	///   however the define should be prefferred when you want to have the option of turning off all logging with the
	///   tb_disable_debug_set define in tb_configuration.h
	///
	#define tb_log_close() TurtleBrains::Debug::CloseLog()

	///
	///	@details Writes a message into the log, the message can be a formatted string behaving exactly like printf would.
	///   Underneath this simply calls the Debug::CloseLog() function, however the define should be prefferred when you 
	///   want to have the option of turning off all logging with the tb_disable_debug_set define in tb_configuration.h
	///
	/// @param formattedMessage A string of characters of a formatted string that describes the variable arguments that
	///   follow.  The formatted message and arguments are used exactly like printf() would be.
	/// @param ... The variable arguments the fill in the formattedMessage string using the same printf() format.
	///
	#define tb_log(formattedMessage, ...) TurtleBrains::Debug::Log(formattedMessage, ##__VA_ARGS__)

	///
	///	@details Writes a message into the log, the message can be a formatted string behaving exactly like printf would,
	///   but does so only if the testResult is true.  If test result is false no message gets displayed on stdout or the log.
	///   Underneath this simply calls the Debug::CloseLog() function, however the define should be prefferred when you 
	///   want to have the option of turning off all logging with the tb_disable_debug_set define in tb_configuration.h
	///
	/// @param testResult An expression that will result in a boolean, true or false, result.  If true the message will
	///   output in the log and stdout, otherwise it will not be displayed.
	/// @param formattedMessage A string of characters of a formatted string that describes the variable arguments that
	///   follow.  The formatted message and arguments are used exactly like printf() would be.
	/// @param ... The variable arguments the fill in the formattedMessage string using the same printf() format.
	///
	#define tb_log_if(testResult, formattedMessage, ...) TurtleBrains::Debug::LogIf((testResult), formattedMessage, ##__VA_ARGS__)

#else	//NO tb_use_debug_set

	#define tb_log_open(logFile, createConsoleWindow) ;
	#define tb_log_save() ;
	#define tb_log_close() ;
	#define tb_log(formattedMessage, ...) ;
	#define tb_log_if(test, formattedMessage, ...) ;

#endif /* tb_use_debug_set */

///
/// @}
///

namespace TurtleBrains
{
	namespace Debug
	{

		///
		///	@details Opens a file to output the log at, and can, on Windows building with Visual CPP, open a console
		///   window for showing std out if the console parameter is passed as true.
		///
		/// @param logFile The filepath of the location to save the log file.  This file will be cleared when loaded so the
		///   contents will only show a single run.
		/// @param createConsoleWindow Set to true this will attempt to create a console window.  Currently only supported on Windows
		///   when building with Visual CPP.
		///
		/// @note Prefer use of the macro tb_log_open so that nothing will happen, compiled out, if debugging is
		///   disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		void OpenLog(const std::string& logFile, bool createConsoleWindow);

		///
		///	@details Saves the log file to disk to make sure any log messages have been flushed.  
		///
		/// @note Prefer use of the macro tb_log_save so that nothing will happen, compiled out, if debugging is
		///   disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		void SaveLog(void);

		///
		///	@details Closes the log file after flushing any pending log messages, once closed any new log messages will not
		///   be written to the log file, only to the stdout.
		///
		/// @note Prefer use of the macro tb_log_close so that nothing will happen, compiled out, if debugging is
		///   disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		void CloseLog(void);

		///
		///	@details Writes a message into the log, the message can be a formatted string behaving exactly like printf would.
		///
		/// @param formattedMessage A string of characters of a formatted string that describes the variable arguments that
		///   follow.  The formatted message and arguments are used exactly like printf() would be.
		/// @param ... The variable arguments the fill in the formattedMessage string using the same printf() format.
		///
		/// @note Prefer use of the macro tb_log so that nothing will happen, compiled out, if debugging is
		///   disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		void Log(const char* formattedMessage, ...);

		///
		///	@details Writes a message into the log, the message can be a formatted string behaving exactly like printf would,
		///   but does so only if the testResult is true.  If test result is false no message gets displayed on stdout or the log.
		///
		/// @param testResult An expression that will result in a boolean, true or false, result.  If true the message will
		///   output in the log and stdout, otherwise it will not be displayed.
		/// @param formattedMessage A string of characters of a formatted string that describes the variable arguments that
		///   follow.  The formatted message and arguments are used exactly like printf() would be.
		/// @param ... The variable arguments the fill in the formattedMessage string using the same printf() format.
		///
		void LogIf(const bool testResult, const char* formattedMessage, ...);

	}; /* namespace Debug */
}; /* namespace TurtleBrains */

namespace tbDebug = TurtleBrains::Debug;

#endif /* _TurtleBrains_DebugLogger_h_ */
