///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_EntityBehaviorInterface_h_
#define _TurtleBrains_EntityBehaviorInterface_h_

#include "tb_entity.h"
#include "../core/tb_noncopyable.h"
#include "../game/tb_game_timer.h"

namespace TurtleBrains
{
	namespace Game
	{

		///
		/// @details The EntityBehaviorInterface provides an interface to create small behavioral states for an Entity
		///   that can be reused and applied on the Entity's behavior stack.
		///
		class EntityBehaviorInterface : public tbCore::Noncopyable
		{
		public:

			///
			/// @details Constructs an entity behavior.
			///
			explicit EntityBehaviorInterface(Entity& entity);

			///
			///	@details Destructs an entity behavior which currently (TurtleBrains v0.2.0) does not have any resources to
			///   clean up after.
			///
			virtual ~EntityBehaviorInterface(void) = 0;

		protected:
			///
			///	@details The Entity which the behavior should work magic on. Some behaviors may need a more specific Entity
			///   reference and that should be held and contained. For instance a behavior specific to a PlayerEntity would
			///   contain PlayerEntity& mPlayerEntity; and work the specific magic on the player.
			///
			Entity& mEntity;

			///
			/// @details This is invoked when the behavior is pushed onto the entities behavior stack, pausing the currently
			///   active behavior. This behavior is now the active behavior and will be paused if that changes.
			///
			virtual void OnAdded(void);

			///
			/// @details This is invoked just as the behavior is being removed from the entity stack.
			///
			virtual void OnRemoved(void);

			///
			/// @details This is invoked when the behavior is currently active and a new behavior gets pushed on becoming the
			///   new active behavior. OnResume() will be invoked once the behavior becomes active once again.
			///
			virtual void OnPaused(void);

			///
			/// @details This is invoked when the behavior is currently paused and the active behavior gets popped off the
			///   stack letting this behavior become active again.
			///
			virtual void OnResume(void);

			///
			/// @details This is invoked when the Entity is activated and simulated and when the behavior is at the top of the
			///   entities behavior stack. Called 0 to N times per frame.
			///
			virtual void OnSimulate(void);

			///
			/// @details This is invoked when the Entity is activated and updated and when the behavior is at the top of the
			///   entities behavior stack. Called exactly once per frame.
			///
			virtual void OnUpdate(const float deltaTime);

			///
			/// @details This is invoked when the Entity is visible and rendered and when the behavior is at the top of the
			///   entities behavior stack. Called exactly once per frame.
			///
			virtual void OnRender(void) const;

			///
			/// @details This is invoked when the Entity collides/intersects another entity and the behavior is at the top of
			///   the entities behavior stack.
			///
			virtual void OnCollideWith(const Entity& other);

		private:
			friend class Entity;
		};

		////// TEMPORARY CRAP FOLLOWS BELOW /////

		///
		///	@details The DelayBehavior is a simple behavior that, while active, counts down a delay timer and pops the
		///   behavior off the entity's behavior stack once the timer reaches zero.
		///
		class DelayBehavior : public EntityBehaviorInterface
		{
		public:
			///
			///	@details Constructs a DelayBehavior for the entity.
			///
			/// @param entity The entity which this behavior will be "controlling".
			/// @param delayTimer The amount of time to delay before popping the action off the entities stack.
			///
			DelayBehavior(Entity& entity, const tbGame::GameTimer& delayTimer) :
				EntityBehaviorInterface(entity),
				mDelayTimer(delayTimer)
			{
			}

			///
			///	@details Destructs the DelayBehavior.
			///
			virtual ~DelayBehavior(void)
			{
			}

			///
			///	@details Decrements the delay timer and if it reaches zero, pops the behavior off the entity's behavior stack.
			///
			virtual void OnSimulate(void);
			
		private:
			tbGame::GameTimer mDelayTimer;
		};


		///
		///	@details The KillBehavior is a simple behavior that will remove the entity from entity manager that it exists
		///   within. Note this does not delete the entity, although the entity manager may depending on how the entity
		///   was added.
		///
		class KillBehavior : public EntityBehaviorInterface
		{
		public:
			///
			///	@details Constructs a KillBehavior for a specific entity.
			///
			/// @param entity The entity that will be removed from the EntityManager it belongs to on the first call to Simulate.
			///
			KillBehavior(Entity& entity) :
				EntityBehaviorInterface(entity)
			{
			}

			///
			///	@details Destructs the KillBehavior.
			///
			virtual ~KillBehavior(void)
			{
			}

			///
			///	@details Removes the entity from the EntityManager it belongs to.
			///
			virtual void OnSimulate(void);

		private:
		};


		///
		///	@details The MoveBehavior is a behavior that will move an Entity at a constant rate and direction.
		///
		/// @note This behavior never removes itself, so if used, the entity should contain logic to warp or kill itself.
		///
		class MoveBehavior : public EntityBehaviorInterface
		{
		public:
			///
			///	@details Creates a MoveBehavior that will move by the rate and direction specified by the velocity vector.
			///
			/// @param entity The entity that will be controlled and moving during this behavior.
			/// @param velocity The direction and speed which the entity will be moving at, by default this speed is in
			///   pixels per second unless the projection or view matrix has changed.
			///
			/// @note This behavior never removes itself, so if used, the entity should contain logic to warp or kill itself.
			///
			MoveBehavior(Entity& entity, const tbMath::Vector2& velocity) :
				EntityBehaviorInterface(entity),
				mVelocity(velocity)
			{
			}

			///
			///	@details Cleans up after the MoveBehavior, which doesn't involve a whole lot.
			///
			virtual ~MoveBehavior(void)
			{
			}

			///
			///	@details Moves the entity by the velocity within the given fixed-time step.
			///
			virtual void OnSimulate(void);

		private:
			tbMath::Vector2 mVelocity;
		};


		///
		///	@details The MoveToBehavior will move an Entity at a constant rate of speed towards a desiredPosition until
		///   they reach that location. The speed will not increase or decrease.
		///
		class MoveToBehavior : public EntityBehaviorInterface
		{
		public:
			///
			///	@details Creates a MoveToBehavior for the entity to move to a desired location, moving at the given speed.
			///
			/// @param entity The entity that will be controlled by the behavior and moved.
			/// @param desiredPosition The location the entity will travel toward, directly.
			/// @param constantMoveSpeed The speed at which the entity will move at during the behavior, without speeding up
			///   or slowing down.
			///
			MoveToBehavior(Entity& entity, const tbMath::Vector2& desiredPosition, float constantMoveSpeed) :
				EntityBehaviorInterface(entity),
				mDesiredPosition(desiredPosition),
				mConstantMoveSpeed(constantMoveSpeed)
			{
			}

			///
			///	@details Cleans up after the MoveToBehavior, which doesn't involve a whole lot.
			///
			virtual ~MoveToBehavior(void)
			{
			}

			///
			///	@details Moves the entity toward the desired position by the constant speed, if the entity would have passed
			///   on by the desired position it stops at the desired position and this behavior will get popped off the entity.
			///
			virtual void OnSimulate(void);

		private:
			tbMath::Vector2 mDesiredPosition;
			float mConstantMoveSpeed;
		};

	}; /* namespace Game */
}; /* namespace TurtleBrains */

namespace tbGame = TurtleBrains::Game;

#endif /* _TurtleBrains_EntityBehaviorInterface_h_ */
