///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tb_entity_manager.h"
#include "../math/unstable/tbu_bounding_volume.h"

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::~EntityManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::AddEntity(Entity* entity)
{
	ReallyAddEntity(entity, true);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::AddEntity(Entity& entity)
{
	ReallyAddEntity(&entity, false);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::ReallyAddEntity(Entity* entity, bool managed)
{
	tb_error_if(nullptr != entity->mEntityManager, "tbExternalError: Cannot add this entity to an entity manager, it is already being managed.");

	if (true == managed)
	{
		mManagedEntities.push_back(entity);
	}

	mEntities.push_back(entity);

	EntityTypeContainer emptyContainer;
	EntityTypeChanged(*entity, emptyContainer);

	AddGraphic(*entity);

	entity->mEntityManager = this;
	entity->OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::RemoveEntity(Entity* entity)
{
	mEntitiesToRemove.insert(entity);

	//auto returnValue = mEntitiesToRemove.insert(entity);
	//tb_warning_if(true == returnValue.second, "tbExternalWarning: Entity is already marked for removal.");
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::RemoveEntities(const EntityType& byType)
{
	if (Entity::kInvalidType == byType)
	{
		mEntitiesToRemove.insert(mEntities.begin(), mEntities.end());
	}
	else
	{
		EntityList& entities(mEntitiesByType[byType]);
		mEntitiesToRemove.insert(entities.begin(), entities.end());
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::SafeToRemoveEntities(void)
{
	for (EntitySet::iterator removeItr = mEntitiesToRemove.begin(), removeEnd = mEntitiesToRemove.end(); removeItr != removeEnd; ++removeItr)
	{
		Entity* entityToRemove = *removeItr;

		RemoveGraphic(entityToRemove);

		mEntities.remove(entityToRemove);
		for (EntityType type : entityToRemove->mEntityTypes)
		{
			mEntitiesByType[type].remove(entityToRemove);
		}

		entityToRemove->mEntityManager = nullptr;
		entityToRemove->OnRemoved();

		//If the entityToRemove is found in the mManagedEntities container, it will need to be deleted.
		for (EntityList::iterator managedItr = mManagedEntities.begin(), managedEnd = mManagedEntities.end(); managedItr != managedEnd; ++managedItr)
		{
			if ((*managedItr) == entityToRemove)
			{
				mManagedEntities.remove(entityToRemove);
				delete entityToRemove;
				break;
			}
		}
	}

	mEntitiesToRemove.clear();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::EntityTypeChanged(Entity& entity, const EntityTypeContainer& oldTypes)
{
	//Remove the Entity from the EntitiesByType container for all oldTypes.
	for (EntityTypeContainer::const_iterator itr = oldTypes.begin(), itrEnd = oldTypes.end(); itr != itrEnd; ++itr)
	{
		mEntitiesByType[*itr].remove(&entity);
	}

	//Add the Entity to the EntitiesByType container for all oldTypes.
	const EntityTypeContainer& newTypes = entity.GetEntityTypes();
	for (EntityTypeContainer::const_iterator itr = newTypes.begin(), itrEnd = newTypes.end(); itr != itrEnd; ++itr)
	{
		mEntitiesByType[*itr].push_back(&entity);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityList tbGame::EntityManager::GetAllEntities(void)
{
	return mEntities;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityList tbGame::EntityManager::GetEntitiesByType(const EntityType& byType)
{
	EntityByTypeMap::iterator itr = mEntitiesByType.find(byType);
	if (itr == mEntitiesByType.end())
	{
		return EntityList();
	}
	return itr->second;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityList tbGame::EntityManager::GetEntitiesAt(const tbMath::Vector2& point, const EntityType& byType,
	bool onlyCollidableEntities)
{
	EntityList returnList;
	EntityList emptyEntityList;
	EntityByTypeMap::iterator byTypeIter;
	EntityList& entities((Entity::kInvalidType == byType) ? mEntities :
		((byTypeIter = mEntitiesByType.find(byType)) == mEntitiesByType.end()) ? emptyEntityList : byTypeIter->second);

	for (EntityList::iterator itr = entities.begin(), itrEnd = entities.end(); itr != itrEnd; ++itr)
	{
		Entity* entity = *itr;

		if (true == entity->IsCollidable() || false == onlyCollidableEntities)
		{
			if (true == entity->CheckCollisionWith(point))
			{
				returnList.push_back(entity);
			}
		}
	}

	return returnList;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityList tbGame::EntityManager::GetEntitiesWithin(const tbMath::Vector2& center, const float radius,
	const EntityType& byType, bool onlyCollidableEntities)
{
	EntityList returnList;
	EntityList emptyEntityList;
	EntityByTypeMap::iterator byTypeIter;
	EntityList& entities((Entity::kInvalidType == byType) ? mEntities :
											 ((byTypeIter = mEntitiesByType.find(byType)) == mEntitiesByType.end()) ? emptyEntityList : byTypeIter->second);

	for (EntityList::iterator itr = entities.begin(), itrEnd = entities.end(); itr != itrEnd; ++itr)
	{
		Entity* entity = *itr;

		if (true == entity->IsCollidable() || false == onlyCollidableEntities)
		{
			if (true == entity->CheckCollisionWith(center, radius))
			{
				returnList.push_back(entity);
			}
		}
	}

	return returnList;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager::EntityList tbGame::EntityManager::GetEntitiesWithin(const tbMath::Vector2& center,
	const float width, const float height, const EntityType& byType, bool onlyCollidableEntities)
{
	EntityList returnList;
	EntityList emptyEntityList;
	EntityByTypeMap::iterator byTypeIter;
	EntityList& entities((Entity::kInvalidType == byType) ? mEntities :
											 ((byTypeIter = mEntitiesByType.find(byType)) == mEntitiesByType.end()) ? emptyEntityList : byTypeIter->second);

	for (EntityList::iterator itr = entities.begin(), itrEnd = entities.end(); itr != itrEnd; ++itr)
	{
		Entity* entity = *itr;

		if (true == entity->IsCollidable() || false == onlyCollidableEntities)
		{
			if (true == entity->CheckCollisionWith(center, width, height))
			{
				returnList.push_back(entity);
			}
		}
	}

	return returnList;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::Simulate(void)
{
	SafeToRemoveEntities();

	for (EntityList::iterator entityItr = mEntities.begin(), entityItrEnd = mEntities.end(); entityItr != entityItrEnd; ++entityItr)
	{
		(*entityItr)->Simulate();
	}

	for (EntityList::iterator outerItr = mEntities.begin(), entityEnd = mEntities.end(); outerItr != entityEnd; ++outerItr)
	{
		Entity* outerEntity(*outerItr);
		if (false == outerEntity->IsCollidable())
		{
			continue;
		}

		EntityList::iterator innerItr = outerItr;
		++innerItr;

		for ( ; innerItr != entityEnd; ++innerItr)
		{
			Entity* innerEntity(*innerItr);
			if (false == innerEntity->IsCollidable())
			{
				continue;
			}

			tb_error_if(innerEntity == outerEntity, "tbInternalError: Should never do collision against the same entity.");

			if (true == outerEntity->CheckCollisionWith(*innerEntity))
			{
				outerEntity->OnCollideWith(*innerEntity);
				innerEntity->OnCollideWith(*outerEntity);
			}
		}
	}

	SafeToRemoveEntities();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::EntityManager::OnUpdate(const float deltaTime)
{
	SafeToRemoveEntities();

	GraphicList::OnUpdate(deltaTime);

	SafeToRemoveEntities();
}

//--------------------------------------------------------------------------------------------------------------------//
