///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_GameSceneInterface_h_
#define _TurtleBrains_GameSceneInterface_h_

#include "tb_entity_manager.h"
#include "../graphics/tb_graphic_list.h"
#include "../graphics/tb_camera.h"

#include <list>

namespace TurtleBrains
{
	namespace Game
	{

		///
		///	@details The GameScene object can also be considered a game state, or screen which contains the logic of a
		///   small part of the game.  Typically many GameScene will build the game, perhaps a TitleScene, OptionsScene,
		///   GameplayScene and CreditsScene.  The GameScene provides an interface to override the behaviors as desired.
		///
		class GameScene : public EntityManager
		{
		public:

			///
			///	@details Initializes the accumulated simulation time to 0 and waits for OnOpen() to be called to 
			///   activate the GameScene.
			///
			GameScene(void);

			///
			///	@details If the active GameScene is the scene being destructed the active scene will be changed to a
			///   blank scene, otherwise does nothing as the object is destructed.
			///
			virtual ~GameScene(void) = 0;

			///
			///	@details Changes the active scene to the gameScene provided, immediately at the end of the current 
			///   frame without any transition or effects.  The current active scene, if one exists, will be closed 
			///   and OnClose() will be invoked before the new gameScene is opened with OnOpen() and set as the active
			///   scene.
			///
			/// @param gameScene The new scene that will be updated and displayed as the active scene.  The scene must
			///   remain in scope for the entire duration while it is active.  TODO: TIM: DocLink: Add a sample about
			///   creating a SceneManager and provide a template.
			///
			static void ChangeToScene(GameScene& gameScene);

			///
			///	@details If there has been a scene change this will actually call the OnClose() of the current scene and
			///   OnOpen() of the pending new scene.  It will then continue to perform a single frame for the scene
			///   by first calling OnSimulate() zero to N times, then calling OnUpdate and OnRender() exactly once for
			///   the active scene.
			///
			/// @param deltaTime The amount of time since the last call to PerformFrame(), should already be capped if
			///   framerate is too low and deltaTime gets too large.
			///
			static void PerformFrame(const float deltaTime);

			///
			///	@details Perform a single frame for the provided gameScene by first calling OnSimulate() zero to N times,
			///   then calling OnUpdate and OnRender() exactly once for the provided scene.
			///
			/// @param gameScene The gameScene to simulate, update and render for a single frame.
			/// @param deltaTime The amount of time since the last call to PerformFrame(), should already be capped if
			///   framerate is too low and deltaTime gets too large.
			///
			static void PerformFrame(GameScene& gameScene, const float deltaTime);
			
		protected:
			///
			///	@details Called zero to N times per frame as simulated time accumulates.  This is where game logic
			///   should live as OnSimulate() is called at a fixed rate.  By default will be called at 100hz, 100
			///   times per second with a fixed time step of 10ms or 0.01f seconds.  This rate can be changed with
			///   the SetSimulateStepsPerSecond() function.
			///
			virtual void OnSimulate(void);

			///
			///	@details Called once per frame every frame to check for any Input events or update things that are
			///   dependent on the frame / wall timer.  Game logic does not generally belong in the Update function
			///   as OnSimulate() is the fixed time step for more deterministic gameplay.
			///
			/// @param deltaTime The number of seconds that has passed on the wall clock since the last frame called
			///   OnUpdate().  This value will be capped, so if the framerate is really bad deltaTime could be lower
			///   than the actual wall-time.
			///
			virtual void OnUpdate(const float deltaTime);

			///
			/// @details This will be called once per frame every frame to display the current frame of the scene.
			///
			virtual void OnRender(void) const;

			///
			///	@details This will be called just before the scene is actually activated so that any resources can be
			///   created or aquired for the scene to display correctly.  This OnOpen() should be called to reset the
			///   accumulated simualtion time just in case there had been some time remaining from a previous activation.
			///
			virtual void OnOpen(void);

			///
			///	@details This will be called when the scene has ended and is no longer active in any manner, all
			///   resources should be cleaned up until the scene is opened again.
			///
			virtual void OnClose(void);

		private:
			float mAccumulatedSimulationTime;
			tbGraphics::Camera mCamera;
		};

	}; /* namespace Game */
}; /* namespace TurtleBrains */

namespace tbGame = TurtleBrains::Game;

#endif /* _TurtleBrains_GameSceneInterface_h_ */
