///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_AnimatedSprite_h_
#define _TurtleBrains_AnimatedSprite_h_

#include "tb_sprite.h"
#include "tb_sprite_map.h"
#include <vector>
#include <map>

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		///	@details This is an ordered sequence of SpriteFrame objects that make up an animation for the AnimatedSprite to
		///   then flip from frame to frame.
		///
		/// @note May eventually contain desired flip rate, default looping / pingpong values etc.
		///
		class AnimationSequence
		{
		public:

			///
			///	@details Creates an empty AnimationSequence with no frames.  Frames must then be added to the sequence in the
			///   order desired by calling AddFrame().
			///
			/// @param textureHandle All frames added to the AnimationSequence must be on this texture.
			///
			/// @note This will trigger an error condition it textureHandle is kInvalidTexture.
			///
			explicit AnimationSequence(const TextureHandle& textureHandle);

			///
			///	@details Creates an AnimationSequence with a custom sequence of mapped frames by breaking down the texture into
			///   columns and rows based on width/height of the frame and texture, then uses the frames parameter as a
			///   sequence of indices into that space.  A 512x256 texture, with a 32x32 frame will have 16 columns and 8 rows.
			///   The index 0 is for the top-left of the texture, the maximum index is (columns * rows) - 1 for the bottom-right
			///   most frame of the texture.
			///
			/// @param spriteMap The SpriteMap object describing the location and framesize of each frame in the animation.
			/// @param frames The customized sequence of indices on the mapped texture columns by rows, starting with top-left
			///   ending at the bottom-right.
			///
			AnimationSequence(const SpriteMap& spriteMap, const std::vector<size_t> frames);

			///
			///	@details Creates an AnimationSequence with an ordered sequence of mapped frames by breaking down the texture into
			///   columns and rows based on width/height of the frame and texture, then uses the frames parameter as a
			///   sequence of indices into that space.  A 512x256 texture, with a 32x32 frame will have 16 columns and 8 rows.
			///   The index 0 is for the top-left of the texture, the maximum index is (columns * rows) - 1 for the bottom-right
			///   most frame of the texture.
			///
			/// @param spriteMap The SpriteMap object describing the location and framesize of each frame in the animation.
			/// @param startFrameIndex The first frame of the ordered sequence.
			/// @param frameCount The number of frames in the sequence, the final frame index will be startFrame + frameCount.
			///
			AnimationSequence(const SpriteMap& spriteMap, const size_t& startFrameIndex, const size_t& frameCount);

			///
			///	@details Cleans up after the AnimationSequence by removing the sequence of SpriteFrame objects from memory.
			///
			~AnimationSequence(void);

			///
			/// @details Adds a single custom frame to the end of the sequence.  Each frame will be played in the order added
			///   to the sequence.  Call AddFrame(); in the order desired.  It is possible to add the same frame multiple
			///   times as well for any desired effects.
			///
			/// @param frame The SpriteFrame object of the next frame in the animation sequence.
			///
			void AddFrame(const SpriteFrame& frame);

			///
			/// @details Builds a SpriteFrame to add to the end of the sequence.  Each frame will be played in the order added
			///   to the sequence.  Call AddFrame(); in the order desired.  It is possible to add the same frame multiple
			///   times as well for any desired effects.
			///
			/// @param frameX The horizontal pixel location of the left edge of the sprite frame, 0 is the left edge of texture.
			/// @param frameY The vertical pixel location of the top edge of the sprite frame, 0 is the top edge of texture.
			/// @param frameWidth The width in pixels of the sprite frame.
			/// @param frameHeight The height in pixels of the sprite frame.
			///
			void AddFrame(const PixelSpace& frameX, const PixelSpace& frameY, const PixelSpace& frameWidth, const PixelSpace& frameHeight);

			///
			///	@details Returns the number of frames in the AnimationSequence.
			///
			size_t GetFrameCount(void) const;

			///
			///	@details Returns the SpriteFrame at the frameIndex or triggers an error condition if frameIndex is out-of-range.
			///
			/// @param frameIndex The index of the SpriteFrame to return, must be in the range: (0 <= frameIndex < GetFrameCount()).
			///
			const SpriteFrame& GetFrame(const size_t& frameIndex) const;

		private:
			//Animation Size: 36 (bytes per frame) * frames + 8 (bytes minimum for this)
			//This could greatly reduce size by hold SpriteFrame references, but added cost of complexity and easy usage.
			TextureHandle mTexture;
			std::vector<SpriteFrame> mAnimationFrames;
		};


		///
		///	@details The AnimatedSprite is a Sprite that contains several AnimationSequences that can be played back to
		///   flip the frames of the sprite to playback an animation.
		///
		class AnimatedSprite : public Sprite
		{
		public:
			///
			///	@details The default amount of time per frame, in seconds by the animation system.  Currently at 30fps
			///   in (TurtleBrains v0.2.0).  This may become an integer type that represents the time in milliseconds.
			///
			/// @note This may become a time in milliseconds with a maximum of 100 frames per second and a minimum of
			///   1 frame per second, although this is yet to be decided.
			///
			static const float& kDefaultTimePerFrame;

			///
			/// @details Sets the default framerate for the AnimatedSprites to use, by default this is set to 30fps. 
			///
			static void SetAnimationFrameRate(const int framesPerSecond);

			///
			/// @details Creates an AnimatedSprite with a given sprite frame, all frames of the sequences added to the
			///   AnimatedSprite must have the same texture handle as this frame or an error condition will be triggered.
			///
			/// @param spriteFrame the initial frame of the Sprite containing the TextureHandle that all other frames
			///   must have in order to be used.
			///
			explicit AnimatedSprite(const SpriteFrame& spriteFrame);

			///
			/// @details Creates an AnimatedSprite similar to AnimatedSprite(tbGraphics::theSpriteManager.GetAnimatedSprite("sheet", "sprite"))
			///   except more convenient.
			///
			/// @param spriteSheetName A name to a SpriteSheet that was loaded during a call to SpriteManager::LoadSpriteSheetFromFile() or
			///   SpriteManager::LoadSpriteSheetFromData(). If no SpriteSheet is found with the provided spriteSheetName, an error 
			///   condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded. If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			///
			AnimatedSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName);

			///
			///	@details Copy constructor to create an animated sprite by copying all sequences from the other object.
			///
			/// @param other The AnimatedSprite to copy and mimic.
			///
			AnimatedSprite(const AnimatedSprite& other);

			///
			///	@details Assignment operator for the AnimatedSprite, this will clear any current animation and sequences,
			///   copy all the sequences from the other object and set the current frame and timers to that other object.
			///
			/// @param other The AnimatedSprite to copy and mimic.
			///
			AnimatedSprite& operator=(const AnimatedSprite& other);

			///
			///	@details Destructs the AnimatedSprite object cleaning up the sequences of animations that had been added.
			///
			virtual ~AnimatedSprite(void);

			///
			///	@details Adds a set of frames to the AnimatedSprite so it can be played as an animation using PlayAnimation().
			///   It is expected that no AnimationSequence has been added with sequenceName and that the sequence has no more
			///   than 256 frames of animation or an error condition will be triggered.
			///
			/// @param sequenceName A name for the animation sequence added so it can be reference later in PlayAnimation
			///   using the same name.  Must not be an empty string or an error condition will occur, must be different
			///   than any other sequence that has been added to the animated sprite or an error condition will be triggered.
			/// @param sequence The sequence of frames that represents the animation.
			///
			void AddSequence(const tbCore::tbString& sequenceName, const AnimationSequence& sequence);

			///
			///	@details Creates an AnimationSequence with the parameters given, literally calling the appropriate
			///   constructor.  This may be removed from the interface in future versions as it is redundant now.
			///   Should favor the use of loading json Sprite Sheets in the SpriteManager with animation and sprite data.
			///
			/// @param sequenceName A name for the animation sequence added so it can be reference later in PlayAnimation
			///   using the same name.  Must not be an empty string or an error condition will occur, must be different
			///   than any other sequence that has been added to the animated sprite or an error condition will be triggered.
			/// @param spriteMap The SpriteMap object describing the location and framesize of each frame in the animation.
			/// @param frames is an ordered container of the frames in the sequences, { 0, 1, 5, 3 }.
			///
			void AddMappedSequence(const tbCore::tbString& sequenceName, const SpriteMap& spriteMap, const std::vector<size_t> frames);

			///
			///	@details Creates an AnimationSequence with the parameters given, literally calling the appropriate
			///   constructor.  This may be removed from the interface in future versions as it is redundant now.
			///   Should favor the use of loading json Sprite Sheets in the SpriteManager with animation and sprite data.
			///
			/// @param sequenceName A name for the animation sequence added so it can be reference later in PlayAnimation
			///   using the same name.  Must not be an empty string or an error condition will occur, must be different
			///   than any other sequence that has been added to the animated sprite or an error condition will be triggered.
			/// @param spriteMap The SpriteMap object describing the location and framesize of each frame in the animation.
			/// @param startFrameIndex The first frame of the ordered sequence.
			/// @param frameCount The number of frames in the sequence, the final frame index will be startFrame + frameCount.
			///
			void AddMappedSequence(const tbCore::tbString& sequenceName, const SpriteMap& spriteMap, const size_t& startFrameIndex,
				const size_t& frameCount);

			///
			///	@details Sets the current sequence to the one found by the sequenceName, the sprite is immediately changed to
			///   the first frame of the sequence, or last if played backwards, and the animation begins the frame timers.
			///
			/// @param sequenceName The name of the sequence to play, as it was added in AddSequence().
			/// @param isLooping Set to true to play a looping animation until either StopAnimation() is called, or another
			///   animation is started with PlayAnimation().
			/// @param isForward Set to true to play the animation forward, or false to start from the last frame and play
			///   backward until the first frame is reached.
			/// @param timePerFrame Sets the minimum amount of time that a frame should be displayed for before changing to
			///   the next frame of the animation.  This may be changing to milliseconds in the future, but currently the
			///   value is in seconds where 1.25 is one and a quarter seconds per frame.
			///
			/// @note timePerFrame may become a time in milliseconds with a maximum of 100 frames per second and a minimum of
			///   1 frame per second, although this is yet to be decided.
			///
			void PlayAnimation(const tbCore::tbString& sequenceName, const bool isLooping, const bool isForward = true,
				const float timePerFrame = kDefaultTimePerFrame);

			///
			///	@details Sets the current sequence to the one found by the sequenceName, the sprite is immediately changed to
			///   the first frame of the sequence, or last if played backwards, and the animation begins the frame timers.
			///
			/// @param sequenceName The name of the sequence to play, as it was added in AddSequence().
			///
			void PlayAnimation(const tbCore::tbString& sequenceName);

			///
			///	@details Stops the animation on the current frame and stops the frame timers from being processed further.  No
			///   frames will be switched until PlayAnimation is called again to begin playing.
			///
			/// @note The implementation details may change in the future pending the addition of a Pause(), if Pause() is
			///   introduced, Stop() will stop the animation and set the frame to the first frame of animation.  This is yet
			///   to be decided.  TODO: TIM: Implementation: Figure out this detail before TurtleBrains public release.
			///
			void StopAnimation(void);

			///
			///	@details Check if the animation is currently playing or if it has reached the end and stopped, or been stopped
			///   with a call to StopAnimation(). This will always return true for a looping animation!
			///
			bool IsAnimationPlaying(void) const;

			///
			/// @details Checks if a specific animation sequence matches the current sequence and that it is currently playing.
			///   True will be returned only if the sequence matches AND the animation is in playing state so not stopped or
			///   at the end of a non-looping, otherwise false indicates the sequence does not match or animation stopped.
			///
			/// @param sequenceName The name of the sequence to check if it is the one playing.
			///
			bool IsAnimationPlaying(const tbCore::tbString& sequenceName) const;

			///
			///	@details Returns the index of the current frame. This function may be removed or changed in future versions
			///   of the framework.
			///
			/// @note This was added during LD32 as a quick hack and may or may not be removed. Use at your own risk.
			///
			size_t GetFrameIndex(void) const { return mCurrentFrameIndex; }

		protected:

			///
			///	@details If the animation is playing this will update the frame timers and swap to the next frame if the timer
			///   has exceeded the maximum time allowed per frame.
			///
			virtual void OnUpdate(const float deltaTime) override;

		private:
			typedef std::map<tbCore::tbString, AnimationSequence> AnimationContainer;
			AnimationContainer mAnimationSequences;

			const AnimationSequence* mCurrentSequence;
			size_t mCurrentFrameIndex;
			float mTimePerFrame;
			float mFrameTimer;
			bool mIsPlaying;
			bool mIsLooping;
			bool mIsForward;

			//The above could become the following for implementation to save space if it ever becomes a concern, will aim
			//for mobile at some point with a few, reasonable? restrictions: Animation speed must be 1fps or faster and no
			//animation can contain more than 256 frames.
			//
			// 4 bits: playing, looping, forward, pingpong
			// 8 bits: currentFrameIndex (0-255)
			// 10 bits: timePerFrame (milliseconds)
			// 10 bits: frameTimer (milliseconds)
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_AnimatedSprite_h_ */


/*

 The AnimatedSprite was designed for thse following usage patterns:

 //Essentially just load it up in a table and create an animated sprite.  (data driven, most likely use case)
 void FooSimple(void)
 //Initialization
 mPlayerSprite = tbGraphics::theSpriteManager.GetAnimatedSprite("player_sheet", "player");

 //Usage
 mPlayerSprite.PlayAnimation("idle"); / LoopAnimation("idle")
 mPlayerSprite.StopAnimation();


 //Must support developer creating sprite frames themeselves:

 void FooMinimal(void)
 //Initialization
 mPlayerSprite = tbGraphics::theSpriteManager.GetSprite("player_sheet", "player");
 AnimationSequence idleSequence();
 //idleSequence.AddFrame(locationX, locationY, width, height);
 idleSequence.AddFrame(0, 0, 32, 32);
 idleSequence.AddFrame(32, 0, 32, 32);
 idleSequence.AddFrame(64, 0, 32, 32);
 idleSequence.AddFrame(0, 32, 64, 64);  //This method allows size and location differences.
 mPlayerSprite.AddSequence("idle", idleSequence);

 AnimationSequence shootSequence;
 // blah blah, shootSequence.AddFrames()
 mPlayerSprite.AddSequence("shoot", shootSequence);

 //Usage
 mPlayerSprite.PlayAnimation("idle"); / LoopAnimation("idle")
 mPlayerSprite.StopAnimation();



 void FooBetterMin(void)
 //Initialization
 mPlayerSprite = GetSprite("player_sheet", "player");
 //Create a sequence of 3 - 32x32 frames starting at frame 0 on the sheet.

 //Too much effort
 //  AnimationSequence idleSequence();
 //  idleSequence.AddMappedFrames();
 //	mPlayerSprite.AddSequence("idle", idleSequence);

 mPlayerSprite.AddSequence("idle", tbGraphics::CreateMappedAnimationSequence(32, 32, 0, 3));
 //mPlayerSprite.AddSequence("shoot", AnimationSequence(32, 32, { 0, 1, 3, 4 }));
 mPlayerSprite.AddMappedSequence("shoot", 32, 32, { 0, 1, 3, 4});

 //Usage
 mPlayerSprite.PlayAnimation("idle"); / LoopAnimation("idle")
 mPlayerSprite.StopAnimation();
 */
