///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Color_h_
#define _TurtleBrains_Color_h_

#include "../core/tb_types.h"

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		///	@details Stores a 24 bit color with 8 bit alpha and provides an interface to set the color through a variety
		///   of inputs either floating point channels or unsigned byte channels.
		///
		class Color
		{
		public:
			//TODO: TIM: Planning: More color definitions? Yellow, Purple, Light Gray, Dark Gray?
			static const Color kWhite;  ///< @details Fully opaque color representing white, ARGB: 0xFFFFFFFF.
			static const Color kBlack;  ///< @details Fully opaque color representing black, ARGB: 0xFF000000.
			static const Color kRed;		///< @details Fully opaque color representing primary red, ARGB: 0xFFFF0000.
			static const Color kGreen;  ///< @details Fully opaque color representing primary green, ARGB: 0xFF00FF00..
			static const Color kBlue;   ///< @details Fully opaque color representing primary blue, ARGB: 0xFF0000FF.

			///
			///	@details Constructs a default Color object with all channels fully on including alpha, resulting in white.
			///
			Color(void);

			///
			///	@details Constructs a Color object from the 32bit color value with each channel as a byte in the order Alpha,
			///   Red, Green, Blue.
			///
			explicit Color(const tbCore::uint32& colorARGB);

			///
			///	@details Construcs a Color object based on floats for each of the channels.  Each channel should be clamped in
			///   the range of 0.0f to 1.0f, although that will happen automatically when setting a color in this function.
			///
			/// @param clampedAlpha The amount of alpha to use for the color from transparent 0.0f to opaque 1.0f.
			/// @param clampedRed   The amount of red to use for the color from none 0.0f to full 1.0f.
			/// @param clampedGreen The amount of green to use for the color from none 0.0f to full 1.0f.
			/// @param clampedBlue  The amount of blue to use for the color from none 0.0f to full 1.0f.
			///
			/// @note This does not normalize the color if any of the channels were out of bounds, it simply clamps them.
			///
			Color(const float clampedAlpha, const float clampedRed, const float clampedGreen, const float clampedBlue);

			///
			///	@details Sets the color value based on the 32bit value with each channel as a byte in the order Alpha, Red,
			///   Green, Blue.
			///
			void SetColor(const tbCore::uint32& colorARGB);

			///
			///	@details Set the color value based on floats for each of the channels.  Each channel should be clamped in the
			///   range of 0.0f to 1.0f, although that will happen automatically when setting a color in this function.
			///
			/// @param clampedAlpha The amount of alpha to use for the color from transparent 0.0f to opaque 1.0f.
			/// @param clampedRed   The amount of red to use for the color from none 0.0f to full 1.0f.
			/// @param clampedGreen The amount of green to use for the color from none 0.0f to full 1.0f.
			/// @param clampedBlue  The amount of blue to use for the color from none 0.0f to full 1.0f.
			///
			/// @note This does not normalize the color if any of the channels were out of bounds, it simply clamps them.
			///
			void SetColor(const float clampedAlpha, const float clampedRed, const float clampedGreen, const float clampedBlue);

			///
			///	@details Sets a color to be a color in between a startColor and finalColor by linearly interpolating between
			///   the two values.
			///
			/// @param clampedTween The amount of interpolation to be applied from 0.0f to be startColor and 1.0f to be the
			///   finalColor.  Any value in between will use linear interpolatation, so 0.5f is the color halfway between the
			///   startColor and finalColor.
			/// @param startColor A color the start from for the in-between linear interpolation.
			/// @param finalColor A color at the end of the in-between linear interpolation.
			///
			void SetColor(const float clampedTween, const Color& startColor, const Color& finalColor);

			///
			///	@details Returns the 32bit color value with each channel as a byte in the order Alpha, Red, Green, Blue.
			///
			tbCore::uint32 GetColorARGB(void) const;

			///
			/// @details Computes and returns the 32bit color with each channel as a byte in the order Alpha, Blue, Green, Red.
			///
			tbCore::uint32 GetColorABGR(void) const;

			///
			///	@details Returns the byte value of the alpha channel from transparent at 0 to 255 fully opaque.
			///
			tbCore::uint8 GetAlphaByte(void) const;

			///
			///	@details Returns the byte value of the red channel from 0 to 255 for full red.
			///
			tbCore::uint8 GetRedByte(void) const;

			///
			///	@details Returns the byte value of the green channel from 0 to 255 for full green.
			///
			tbCore::uint8 GetGreenByte(void) const;

			///
			///	@details Returns the byte value of the blue channel from 0 to 255 for full blue.
			///
			tbCore::uint8 GetBlueByte(void) const;

			///
			///	@details Returns the clamped value of the alpha channel from transparent at 0.0f to 1.0f fully opaque.
			///
			float GetAlpha(void) const;

			///
			///	@details Returns the clamped value of the red channel from 0.0f to 1.0f representing full red.
			///
			float GetRed(void) const;

			///
			///	@details Returns the clamped value of the green channel from 0.0f to 1.0f representing full green.
			///
			float GetGreen(void) const;

			///
			///	@details Returns the clamped value of the blue channel from 0.0f to 1.0f representing full blue.
			///
			float GetBlue(void) const;

		private:
			//The floats may disappear for size reasons and use GetChannelAsFloat()
			float mAlpha;
			float mRed;
			float mGreen;
			float mBlue;

			tbCore::uint32 mARGB;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Color_h_ */
