///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Graphic_h_
#define _TurtleBrains_Graphic_h_

#include "../math/tb_vector.h"
#include "tb_color.h"
#include "tb_texture_manager.h"

namespace TurtleBrains
{
	namespace Graphics
	{
		///
		/// @details Returns the width of the screen/window as a floating point number.
		///
		float ScreenWidth(void);

		///
		/// @details Returns the height of the screen/window as a floating point number.
		///
		float ScreenHeight(void);

		///
		/// @details Creates and returns a Vector2 object of the location of the center of the screen, half width / height.
		///
		tbMath::Vector2 ScreenCenter(void);

		///
		///	@details These locations are used in the Sprite::SetOrigin() which takes an AnchorLocation.
		///
		enum AnchorLocation
		{
			kAnchorCenter = 0,   ///< This anchor will set the origin to the center of the Sprite horizontally and vertically.
			kAnchorTopLeft,      ///< This anchor will set the origin to the top-left corner, effectively: SetOrigin(0.0f, 0.0f).
			kAnchorTopCenter,    ///< This anchor will set the origin to the top and horizontal center of the Sprite.
			kAnchorTopRight,     ///< This anchor will set the origin to the top and right most corner of the Sprite.
			kAnchorCenterLeft,   ///< This anchor will set the origin to the center vertically and left most edge of the Sprite.
			kAnchorCenterRight,  ///< This anchor will set the origin to the center vertically and right most edge of the Sprite.
			kAnchorBottomLeft,   ///< This anchor will set the origin to the bottom and left most corner of the Sprite.
			kAnchorBottomCenter, ///< This anchor will set the origin to the bottom edge and horizontal center of the Sprite.
			kAnchorBottomRight,  ///< This anchor will set the origin to the bottom and right most corner of the Sprite.
		};

		///
		///	@details The Graphic is an abstract interface for Text, Sprite and other graphical objects that can be
		///   updated and displayed.
		///
		class Graphic
		{
		public:
			///
			///	@details Constructs a Graphic object that starts out active and visible with a position of (0.0f, 0.0f)
			///  and depth at 0.0f.
			///
			Graphic(void);

			///
			/// @details Constructs a Graphic object by copying the details from the other Graphic object.
			///
			/// @param other The Graphic object to copy and mimic.
			///
			Graphic(const Graphic& other);

			///
			///	@details Destroys the Graphic object by doing nothing since no resources need to be cleaned up.
			///
			virtual ~Graphic(void);

			///
			///	@details Returns true if the Graphic is currently active and needing to be updated, or false if the
			///   OnUpdate() calls should be skipped.
			///
			bool IsActive(void) const;

			///
			///	@details Sets a flag that when false causes the Graphic to skip the OnUpdate() calls and when true will
			///   proceed with the OnUpdate() calls.  Defaults to true, may lead to better performance if false when
			///   the object does not need to be updated.
			///
			void SetActive(const bool isActive);

			///
			///	@details First checks to see if the object IsActive() and calls OnUpdate() if it is.  If the object is
			///   not active nothing happens.
			///
			void Update(const float deltaTime);

			///
			///	@details Returns true if the Graphic is currently visible and needing to be displayed, or false if the
			///   OnRender() calls should be skipped.
			///
			bool IsVisible(void) const;

			///
			///	@details Sets a flag that when false causes the Graphic to skip the OnRender() calls and when true will
			///   proceed with the OnRender() calls.  Defaults to true, may lead to better performance if false when
			///   the object does not need to be displayed.
			///
			void SetVisible(const bool isVisible);

			///
			///	@details First checks to see if the object IsVisible() and calls OnRender() if it is.  If the object is
			///   not visible nothing happens.
			///
			void Render(void) const;

			///
			///	@details Returns the position of the sprite as a Vector2 object.
			///
			const tbMath::Vector2& GetPosition(void) const;

			///
			///	@details Sets the position of the object to the x and y location in world space.
			///
			/// @param x The position in the world along the worlds X-axis. 0 being the origin, by default the world
			///   X-axis is horizontal with 0 being the left edge of the screen and positive numbers toward the right.
			/// @param y The position in the world along the worlds Y-axis. 0 being the origin, by default the world
			///   Y-axis is vertical with 0 being the top edge of the screen and positive numbers toward the bottom.
			///
			void SetPosition(const float x, const float y);

			///
			///	@details Sets the position of the object to the location in world space with the given x and y
			///   components of the Vector2.
			///
			/// @param position Where in the world along the each axis the Graphic should be placed. By default the world
			///   X-axis is horizontal with 0 being the left edge of the screen and positive numbers toward the right and
			///   the Y-axis is vertical with 0 being the top edge of the screen and positive numbers toward the bottom.
			///
			void SetPosition(const tbMath::Vector2& position);

			///
			///	@details Sets the origin or a hotspot of the sprite.  This is an offset where the sprite will be
			///   positioned by and rotated around.  The AnchorLocation will describe where the origin offset with
			///   a 9 locations, the four corners, and center points of each edge.
			///
			void SetOrigin(const AnchorLocation& anchor);

			///
			///	@details Sets the origin or a hotspot of the sprite.  This is an offset where the sprite will be
			///   positioned by and rotated around.
			///
			/// @note Defaults to center, may default to top-left in future versions.
			///
			inline void SetOrigin(const float originX, const float originY) { mOrigin.x = originX; mOrigin.y = originY; }

			///
			///	@details Returns the depth, Z value, of the Graphic object.
			///
			float GetDepth(void) const;

			///
			///	@details Sets the depth of the graphic object which should be within a range of -1.0f for objects far
			///   in the background and 1.0f for the objects nearest to the camera.  The default value is 0.0f.
			///
			/// @param newDepth The depth of the Graphic object, smaller being farther away and larger being closer to
			///   the camera.
			///
			void SetDepth(const float newDepth);

			///
			///	@details Changes the scaling of the sprite uniformly on each axis.
			///
			/// @param scale The amount of scaling to apply, 1.0 is normal scale, 0.5 is half size and 2.0 is double.
			///
			inline void SetScale(const float scale) { mScale.x = mScale.y = scale; }

			///
			///	@details Changes the scaling of the sprite on each axis separately.
			///
			/// @param scaleX The amount of scaling to apply on the X-axis, 1.0 is normal scale, 0.5 is half size.
			/// @param scaleY The amount of scaling to apply on the Y-axis, 1.0 is normal scale, 2.0 is double size.
			///
			inline void SetScale(const float scaleX, const float scaleY) { mScale.x = scaleX; mScale.y = scaleY; }

			///
			///	@details Set the rotation/orientation of the sprite so it can rotate around the game world and animate
			///   things such as windmills, tires, etc.
			///
			/// @param rotation The angle in degrees of the sprite, positive values rotate clockwise.
			///
			inline void SetRotation(const float rotation) { mOrientation = rotation; }

			///
			/// @details Returns the current rotation/orientation of the sprite.
			///
			inline float GetRotation(void) const { return mOrientation; }

			///
			///	@details This value changes how the graphic interacts with the camera, a scroll value of 0 will attach the
			///   graphic to the camera which can be useful for UI graphic objects. A scroll value of 1 will scroll at a
			///   normal rate as the camera moves around while other values can create a parallaxing effect.
			///
			inline void SetScroll(const float scroll) { mScroll.x = scroll; mScroll.y = scroll; }

			///
			///	@details This value changes how the graphic interacts with the camera, a scroll value of 0 will attach the
			///   graphic to the camera which can be useful for UI graphic objects. A scroll value of 1 will scroll at a
			///   normal rate as the camera moves around while other values can create a parallaxing effect.
			///
			/// @param scrollX The scroll value for horizontal movement/interaction with the camera.
			/// @param scrollY The scroll value for vertical movement/interaction with the camera.
			///
			inline void SetScroll(const float scrollX, const float scrollY) { mScroll.x = scrollX; mScroll.y = scrollY; }

			///
			///	@details This will return true if the Graphic object is relative to it's parent object, such as the
			///   contained GraphicList.  Currently all Graphics are always relative, although this may be updated
			///   in future versions of TurtleBrains.
			///
			bool IsRelative(void) const;

			//
			//	@details TODO: TIM: Implementation: Currently this is unimplemented and undocumented, all Graphics
			//    are currently relative.
			//
			//void SetRelative(const bool isRelative);

			///
			///	@details Change the color of the sprite so it can fade in/out with alpha, or modify the color for
			///   other effects.
			///
			virtual void SetColor(const tbGraphics::Color& newColor);

			///
			/// @details This should be overridden to return the width of the graphic in PixelSpace units to be used in
			///   placement and anchor point computations.
			///
			virtual PixelSpace GetPixelWidth(void) const { return 0; }

			///
			/// @details This should be overridden to return the height of the graphic in PixelSpace units to be used in
			///   placement and anchor point computations.
			///
			virtual PixelSpace GetPixelHeight(void) const { return 0;  }

			///
			/// @details This is really the width of the graphic object in PixelSpace converted to a floating point for
			///   convenience.
			///
			float GetWidth(void) const { return static_cast<float>(GetPixelWidth()); }

			///
			/// @details This is really the height of the graphic object in PixelSpace converted to a floating point for
			///   convenience.
			///
			float GetHeight(void) const { return static_cast<float>(GetPixelHeight()); }

		protected:
			///
			///	@details This should be overridden by a subclass if it needs to perform any updates per frame before
			///   displaying / rendering a new frame.  This function will be invoked when Update() is called and the
			///   object IsActive().
			///
			/// @note OnUpdate() should not be called directly, even from a subclass.  Use Update() which will first 
			///   check if the Graphic IsActive() before updating with OnUpdate().
			///
			virtual void OnUpdate(const float deltaTime);

			///
			///	@details This must be overridden by a subclass to actually display the Graphic on the screen.  The
			///   function will be invoked when Render() is called and the object IsVisible().
			///
			/// @note OnRender() should not be called directly, even from a subclass.  Use Render() which will first 
			///   check if the Graphic IsVisible() before displaying with OnRender().
			///
			virtual void OnRender(void) const = 0;

			///
			///	@details Returns the color of the Graphic which defaults to Color::kWhite but can be changed
			///   with SetColor().
			///
			const Color& GetColor(void) const;

			///
			/// @details Applies the objects scaling, rotation, camera and translation transform before it gets rendered. This
			///   is automatically called in Render() before OnRender() gets called if the object is visible.
			///
			virtual void ApplyTransform(void) const;

			///
			/// @details Removes all transformations that were applied during ApplyTransform as if nothing even happened. This
			///   is automatically called in Render() after OnRender() was called if the object is visible.
			///
			virtual void PopTransform(void) const;

		private:
			tbMath::Vector2 mPosition;
			tbMath::Vector2 mScroll;
			tbMath::Vector2 mOrigin;	//0,0 = top-left, 1/2w, 1/2h = center.
			tbMath::Vector2 mScale;
			float mOrientation;
			float mDepthZ;
			bool mIsActive;
			bool mIsVisible;
			bool mIsRelative;
			Color mColor;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Graphic_h_ */
