///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Sprite_h_
#define _TurtleBrains_Sprite_h_

#include "tb_graphic.h"
#include "tb_texture_manager.h"
#include "tb_color.h"
#include "../math/tb_math.h"

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		/// @details A sprite frame describes a location and size of a rectangle on a given texture.  Although the members
		///   of this object are public, they should be considered TurtleBrains implementation details and not used as they
		///   are subject to change in the future.
		///
		/// @note This may become a hidden implementation detail in future version of TurtleBrains. Don't construct or use
		///   the SpriteFrame or risk the need to change code.  You've been warned.
		///
		struct SpriteFrame
		{	//TODO: TIM: Planning: These may turn into PixelSpace values, (int), that is what they are, but they are used as float.
			float x;   ///< @details Horizontal location of the SpriteFrame in PixelSpace.  (This is an implementation detail, don't use it.)
			float y;   ///< @details Horizontal location of the SpriteFrame in PixelSpace.  (This is an implementation detail, don't use it.)
			float w;   ///< @details Width of the SpriteFrame in PixelSpace.  (This is an implementation detail, don't use it.)
			float h;   ///< @details Height of the SpriteFrame in PixelSpace.  (This is an implementation detail, don't use it.)
			TexelSpace uvx; ///< @details Horizontal location of the SpriteFrame in TexelSpace.  (This is an implementation detail, don't use it.)
			TexelSpace uvy; ///< @details Vertical location of the SpriteFrame in TexelSpace.  (This is an implementation detail, don't use it.)
			TexelSpace uvw; ///< @details Width of the SpriteFrame in TexelSpace.  (This is an implementation detail, don't use it.)
			TexelSpace uvh; ///< @details Height of the SpriteFrame in TexelSpace.  (This is an implementation detail, don't use it.)
			TextureHandle mTexture;   ///< @details The handle of the texture loaded from the TextureManager that the sprite frame references.  (This is an implementation detail, don't use it.)

			///
			/// @details Returns the pixel space location of the top edge of the sprite frame on the texture.
			///
			PixelSpace GetPixelTop(void) const;

			///
			/// @details Returns the pixel space location of the left edge of the sprite frame on the texture.
			///
			PixelSpace GetPixelLeft(void) const;

			///
			/// @details Returns the pixel space location of the right edge of the sprite frame on the texture.
			///
			PixelSpace GetPixelRight(void) const;

			///
			/// @details Returns the pixel space location of the bottom edge of the sprite frame on the texture.
			///
			PixelSpace GetPixelBottom(void) const;

			///
			/// @details Returns the width of the sprite frame in pixels, also number of pixels from left to right edges.
			///
			PixelSpace GetPixelWidth(void) const;

			///
			/// @details Returns the height of the sprite frame in pixels, also number of pixels from top to bottom edges.
			///
			PixelSpace GetPixelHeight(void) const;

			///
			/// @details Returns the texel space location of the top edge of the sprite frame on the texture.
			///
			TexelSpace GetTexelTop(void) const;

			///
			/// @details Returns the texel space location of the left edge of the sprite frame on the texture.
			///
			TexelSpace GetTexelLeft(void) const;

			///
			/// @details Returns the texel space location of the right edge of the sprite frame on the texture.
			///
			TexelSpace GetTexelRight(void) const;

			///
			/// @details Returns the texel space location of the bottom edge of the sprite frame on the texture.
			///
			TexelSpace GetTexelBottom(void) const;

			///
			/// @details Returns the width of the sprite frame in texel space, also same as texels from left to right edges.
			///
			TexelSpace GetTexelWidth(void) const;

			///
			/// @details Returns the height of the sprite frame in texel space, also same as texels from top to bottom edges.
			///
			TexelSpace GetTexelHeight(void) const;

			///
			///	@details Creates a SpriteFrame object that fills the entire texture from 0,0 to width,height of the texture.
			///
			/// @param textureHandle the handle of the texture to fill with a SpriteFrame.
			///
			static SpriteFrame CreateWith(const TextureHandle& textureHandle);

			///
			///	@details Creates a SpriteFrame object at the desired location on the texture with a size specified.  An error
			///   condition will be triggered if the textureHandle is invalid or if the location/size extends beyond the
			///   boundaries of the textured.
			///
			/// @param textureHandle Must be the handle of a valid texture or an error condition will be triggered.
			/// @param frameX is the location of the left edge of the sprite frame, 0 being the left most edge of the texture.
			/// @param frameY is the location of the top edge of the sprite frame, 0 being the top most edge of the texture.
			/// @param frameWidth is the horizontal size of the sprite frame on the texture, in pixels.
			/// @param frameHeight is the vertical size of the sprite frame on the texture, in pixels.
			///
			static SpriteFrame CreateWith(const TextureHandle& textureHandle, const PixelSpace& frameX, const PixelSpace& frameY,
				const PixelSpace& frameWidth, const PixelSpace& frameHeight);
		};

		///
		///	@details The Sprite is likely the most common Graphic object used when creating a two-dimensional game.
		///   The Sprite contains a little information about the Texture and a rectangle on that texture that describes
		///   what should be rendered by the Sprite.
		///
		class Sprite : public Graphic
		{
		public:
			///
			/// @details Constructs a Sprite object with the given SpriteFrame which contains the texture and location
			///   size of the sprite.
			///
			explicit Sprite(const SpriteFrame& spriteFrame);

			///
			///	@details Construct a sprite from a filepath of a texture with a given frame location and size.
			///
			/// @param textureFile The filepath of a texture to load into the texture manager, if it isn't already, and
			///   use as the texture for the sprites frame.
			/// @param frameX The location of the left edge of the sprites frame in pixels from the left edge of the texture.
			/// @param frameY The location of the top edge of the sprites frame in pixels from the top edge of the texture.
			/// @param frameWidth The width of the sprite frame in pixels, frameWidth + frameX must not exceed texture width,
			///   or an error condition will be triggered.
			/// @param frameHeight The height of the sprite frame in pixels, frameHeight + frameY must not exceed texture height,
			///   or an error condition will be triggered.
			///
			explicit Sprite(const tbCore::tbString& textureFile, const PixelSpace& frameX = 0, const PixelSpace& frameY = 0,
				const PixelSpace& frameWidth = 0, const PixelSpace& frameHeight = 0);

			///
			///	@details Construct a sprite from a handle of an already loaded texture given a frame location and size.
			///
			/// @param textureHandle The handle of the loaded texture, cannot be kInvalidTexture or an error condition
			///   will be triggered.
			/// @param frameX The location of the left edge of the sprites frame in pixels from the left edge of the texture.
			/// @param frameY The location of the top edge of the sprites frame in pixels from the top edge of the texture.
			/// @param frameWidth The width of the sprite frame in pixels, frameWidth + frameX must not exceed texture width,
			///   or an error condition will be triggered.
			/// @param frameHeight The height of the sprite frame in pixels, frameHeight + frameY must not exceed texture height,
			///   or an error condition will be triggered.
			///
			explicit Sprite(const TextureHandle& textureHandle, const PixelSpace& frameX = 0, const PixelSpace& frameY = 0,
											const PixelSpace& frameWidth = 0, const PixelSpace& frameHeight = 0);

			///
			/// @details Creates an Sprite similar to Sprite(tbGraphics::theSpriteManager.GetSprite("sheet", "sprite"))
			///   except more convenient.
			///
			/// @param spriteSheetName A name to a SpriteSheet that was loaded during a call to SpriteManager::LoadSpriteSheetFromFile() or
			///   SpriteManager::LoadSpriteSheetFromData(). If no SpriteSheet is found with the provided spriteSheetName, an error 
			///   condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded. If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			///
			Sprite(const tbCore::tbString& spriteSheet, const tbCore::tbString& spriteName);

			///
			///	@details Construct a sprite from another spriute, copying the sprites frame, color, scaling and other
			///   properties.
			///
			/// @param other The sprite to copy and mimic.
			///
			Sprite(const Sprite& other);

			///
			///	@details Destroys a sprite object, which doesn't do any heavy lifting at this time.
			///
			virtual ~Sprite(void);

			///
			///	@details Set a sprite object to be the show the same sprite frame, color, scaling, origin and all
			///   other properties copied from the other sprite.
			///
			/// @param other The sprite to copy and mimic.
			///
			Sprite& operator=(const Sprite& other);

			///
			/// @TODO: TIM: Documentation: Teach the user how to use this.
			///
			virtual PixelSpace GetPixelWidth(void) const override;

			///
			/// @TODO: TIM: Documentation: Teach the user how to use this.
			///
			virtual PixelSpace GetPixelHeight(void) const override;

		protected:
			///
			///	@details Actually performs the rendering of the sprite to the screen with OpenGL.
			///
			/// @note OnRender() should not be called directly, you should call Render() which will first check if the
			///   Graphic is visible before rendering.
			///
			virtual void OnRender(void) const override;

			///
			///	@details Changes the frame of the Sprite so it can flip to a new frame for AnimatedSprite or other
			///   effects from subclasses.
			///
			/// @note The sprite frame textureHandle MUST match the current sprite frame texture handle or an error condition
			///   will be triggered.
			///
			void SetSpriteFrame(const SpriteFrame& spriteFrame);

			///
			///	@details Returns the current sprite frame to be inspected or used by subclasses.
			///
			const SpriteFrame& GetSpriteFrame(void) const;

		private:
			SpriteFrame mSpriteFrame;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Sprite_h_ */
