///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_sprite_manager.h"
#include "tb_texture_manager.h"
#include "tb_sprite_map.h"

#include "../core/tb_error.h"
#include "../core/tb_opengl.h"
#include "../core/tb_json_parser.h"

#include <fstream>

tbGraphics::SpriteManager tbGraphics::theSpriteManager;

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteManager::SpriteManager(void) :
	mSpriteSheets()
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteManager::~SpriteManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGraphics::SpriteManager::LoadSpriteSheetFromFile(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteSheetFile)
{
	const tbCore::DynamicStructure spriteSheetData(tbCore::LoadJsonFile(spriteSheetFile));

	if (true == spriteSheetData.IsNil() || false == spriteSheetData.IsStructure())
	{
		tb_error("tbDataError: Expected sprite sheet data to be a structure of information.  Sheet: %s", spriteSheetName.c_str());
		return false;
	}

	const tbCore::tbString textureFile = spriteSheetData.GetMember("texture").AsString();
	const TextureHandle& textureHandle = theTextureManager.CreateTextureFromFile(textureFile);
	if (kInvalidTexture == textureHandle)
	{
		tb_error("Was unable to load texture: %s\n", textureFile.c_str());
		return false;
	}

	const int textureWidth(static_cast<int>(theTextureManager.GetTextureWidth(textureHandle)));
	const int textureHeight(static_cast<int>(theTextureManager.GetTextureHeight(textureHandle)));
	const int defaultSpriteWidth(spriteSheetData.GetMember("sprite_width").AsIntegerWithDefault(textureWidth));
	const int defaultSpriteHeight(spriteSheetData.GetMember("sprite_height").AsIntegerWithDefault(textureHeight));

	const tbCore::DynamicStructure& spriteFrames(spriteSheetData.GetMember("sprites"));
	if (true == spriteFrames.IsNil() || false == spriteFrames.IsArray())
	{
		tb_error("tbDataError: Expected sprite sheet to have an array named: sprites. Sheet: \"%s\"", spriteSheetName.c_str());
		return false;
	}

	const tbCore::DynamicStructure& animationSequences(spriteSheetData.GetMember("animations"));
	if (false == animationSequences.IsNil() && false == animationSequences.IsArray())
	{
		tb_error("tbDataError: \"animations\" must be an array if included in sprite object. Sheet: \"%s\"", spriteSheetName.c_str());
		return false;
	}

	SpriteSheet& spriteSheet = mSpriteSheets[spriteSheetName];
	spriteSheet.SetTextureHandle(textureHandle);

	//
	// Load each of the named sprites on the sprite sheet.
	//
	for (size_t frameIndex = 0; frameIndex < spriteFrames.Size(); ++frameIndex)
	{
		const tbCore::DynamicStructure& frameData(spriteFrames[frameIndex]);
		if (true == frameData["name"].IsNil())
		{ //Maybe just make this a warning and discard the frame?
			tb_error("tbDataError: Expected each sprite_frame object to have a name. Sheet: %s", spriteSheetName.c_str());
			return false;
		}

		tbCore::tbString frameName(frameData["name"].AsString());

		const int frameX((false == frameData["x"].IsNil()) ? frameData["x"].AsInteger() : 0);
		const int frameY((false == frameData["y"].IsNil()) ? frameData["y"].AsInteger() : 0);
		const int frameWidth(frameData["width"].AsIntegerWithDefault(defaultSpriteWidth));
		const int frameHeight(frameData["height"].AsIntegerWithDefault(defaultSpriteHeight));

		spriteSheet.AddSpriteFrame(frameName, SpriteFrame::CreateWith(textureHandle, frameX, frameY, frameWidth, frameHeight));

		if (false == frameData["animations"].IsNil())
		{
			const tbCore::DynamicStructure& spriteAnimations(frameData["animations"]);
			if (false == spriteAnimations.IsArray())
			{	//Maybe just make this a warning and disregard.
				tb_error("tbDataError: Expected sprite_frame animations item to be an array if it exists. Sprite \"%s\" on Sheet: \"%s\"", frameName.c_str(), spriteSheetName.c_str());
				return false;
			}
			std::vector<tbCore::tbString> animationNames;
			animationNames.reserve(spriteAnimations.Size());
			for (size_t animationIndex = 0; animationIndex < spriteAnimations.Size(); ++animationIndex)
			{
				animationNames.push_back(spriteAnimations[animationIndex].AsString());
			}

			spriteSheet.AddAnimationSequencesToSprite(frameName, animationNames);
		}
	}

	//
	// Load the animation sequences, either fully custom with named sprite frames, mapped-ordered or mapped-unordered.
	//
	for (size_t animationIndex = 0; animationIndex < animationSequences.Size(); ++animationIndex)
	{
		const tbCore::DynamicStructure& animationData(animationSequences[animationIndex]);
		tb_error_if(true == animationData["name"].IsNil(), "tbDataError: Expected each animation object to have a name. Sheet: %s", spriteSheetName.c_str());
		const tbCore::tbString animationName(animationData["name"]);

		tb_error_if(true == animationData["frames"].IsNil(), "tbDataError: Expected each animation object to have frames. Animation: \"%s\" on Sheet: \"%s\"", animationName.c_str(), spriteSheetName.c_str());
		const tbCore::DynamicStructure& animationFrameData(animationData["frames"]);

		if (true == animationFrameData.IsArray())
		{
			AnimationSequence animationSequence(textureHandle);
			for (size_t frameIndex = 0; frameIndex < animationFrameData.Size(); ++frameIndex)
			{
				tb_error_if(false == animationFrameData[frameIndex].IsString(), "tbDataError: Expected each item in Animation \"%s\" frames array to be a string. Sheet: \"%s\"", animationName.c_str(), spriteSheetName.c_str());
				animationSequence.AddFrame(spriteSheet.GetSpriteFrame(animationFrameData[frameIndex].AsString()));
			}

			spriteSheet.AddAnimationSequence(animationName, animationSequence);
		}
		else if (true == animationFrameData.IsStructure())
		{
			tb_error_if(true == animationFrameData["width"].IsNil(), "tbDataError: Expected each mapped animation object to have a width. Animation: \"%s\" on Sheet: %s", animationName.c_str(), spriteSheetName.c_str());
			tb_error_if(true == animationFrameData["height"].IsNil(), "tbDataError: Expected each mapped animation object to have a height. Animation: \"%s\" on Sheet: %s", animationName.c_str(), spriteSheetName.c_str());
			const int frameWidth(animationFrameData["width"].AsIntegerWithDefault(defaultSpriteWidth));
			const int frameHeight(animationFrameData["height"].AsIntegerWithDefault(defaultSpriteHeight));
			const int locationX(animationFrameData["locationX"].AsIntegerWithDefault(0));
			const int locationY(animationFrameData["locationY"].AsIntegerWithDefault(0));

			const tbCore::DynamicStructure& mappedIndices(animationFrameData["frames"]);
			if (true == mappedIndices.IsNil())
			{
				tb_error_if(true == animationFrameData["start"].IsNil(), "tbDataError: Expected each mapped animation object to have a start frame. Animation: \"%s\" on Sheet: \"%s\"", animationName.c_str(), spriteSheetName.c_str());
				tb_error_if(true == animationFrameData["count"].IsNil(), "tbDataError: Expected each mapped animation object to have a frame count. Animation: \"%s\" on Sheet: \"%s\"", animationName.c_str(), spriteSheetName.c_str());
				const int startFrame(animationFrameData["start"]);
				const int frameCount(animationFrameData["count"]);

				SpriteMap spriteMap(textureHandle, frameWidth, frameHeight, 0, 0, locationX, locationY);
				AnimationSequence mappedSequence(spriteMap, startFrame, frameCount);
				spriteSheet.AddAnimationSequence(animationName, mappedSequence);
			}
			else
			{
				tb_error_if(false == mappedIndices.IsArray(), "tbDataError: Expected mapped animation object with frames item being an array: %s", spriteSheetName.c_str());
				std::vector<size_t> frames;
				frames.resize(mappedIndices.Size());
				for (size_t frameIndex = 0; frameIndex < mappedIndices.Size(); ++frameIndex)
				{
					frames[frameIndex] = mappedIndices[frameIndex].AsInteger();
				}

				SpriteMap spriteMap(textureHandle, frameWidth, frameHeight, 0, 0, locationX, locationY);
				AnimationSequence mappedSequence(spriteMap, frames);
				spriteSheet.AddAnimationSequence(animationName, mappedSequence);
			}
		}
		else
		{
			tb_error("tbDataError: Expected animation frames to be an array or mapped structure. Sheet: %s", spriteSheetName.c_str());
		}
	}
	
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::SpriteManager::GetSpriteSheetTextureHandle(const tbCore::tbString& spriteSheetName) const
{
	SpriteSheetContainer::const_iterator sheetIterator(mSpriteSheets.find(spriteSheetName));
	tb_error_if(sheetIterator == mSpriteSheets.end(), "SpriteSheet with name %s does not exist.", spriteSheetName.c_str());
	return sheetIterator->second.GetTextureHandle();
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteFrame tbGraphics::SpriteManager::GetSpriteFrame(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const
{
	SpriteSheetContainer::const_iterator sheetIterator(mSpriteSheets.find(spriteSheetName));
	tb_error_if(sheetIterator == mSpriteSheets.end(), "SpriteSheet with name %s does not exist.", spriteSheetName.c_str());
	//If spriteName is not found on the spriteSheet an error will be triggered within GetSpriteFrame().
	return sheetIterator->second.GetSpriteFrame(spriteName);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite tbGraphics::SpriteManager::GetSpriteFromFile(const tbCore::tbString& textureFile,
	const PixelSpace& frameX, const PixelSpace& frameY, const PixelSpace& frameWidth, const PixelSpace& frameHeight) const
{
	return Sprite(textureFile, frameX, frameY, frameWidth, frameHeight);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite tbGraphics::SpriteManager::GetSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const
{
	return Sprite(GetSpriteFrame(spriteSheetName, spriteName));
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite tbGraphics::SpriteManager::GetAnimatedSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const
{
	SpriteSheetContainer::const_iterator sheetIterator(mSpriteSheets.find(spriteSheetName));
	tb_error_if(sheetIterator == mSpriteSheets.end(), "SpriteSheet with name %s does not exist.", spriteSheetName.c_str());

	const SpriteSheet& spriteSheet(sheetIterator->second);

	AnimatedSprite animatedSprite(spriteSheet.GetSpriteFrame(spriteName));
	AddAnimationSequences(spriteSheet, spriteName, animatedSprite);
	return animatedSprite;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::SpriteManager::AddAnimationSequences(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName, AnimatedSprite& animatedSprite) const
{
	SpriteSheetContainer::const_iterator sheetIterator(mSpriteSheets.find(spriteSheetName));
	tb_error_if(sheetIterator == mSpriteSheets.end(), "SpriteSheet with name %s does not exist.", spriteSheetName.c_str());

	const SpriteSheet& spriteSheet(sheetIterator->second);
	AddAnimationSequences(spriteSheet, spriteName, animatedSprite);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::SpriteManager::AddAnimationSequences(const SpriteSheet& spriteSheet, const tbCore::tbString& spriteName, AnimatedSprite& animatedSprite) const
{
	if (true == spriteSheet.HasAnimationsForSprite(spriteName))
	{
		auto animationNames = spriteSheet.GetAnimationsForSprite(spriteName);
		for (size_t animationIndex = 0; animationIndex < animationNames.size(); ++animationIndex)
		{
			const tbCore::tbString& animationName(animationNames[animationIndex]);
			const AnimationSequence& animationSequence(spriteSheet.GetAnimationSequence(animationName));
			animatedSprite.AddSequence(animationName, animationSequence);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//
