///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_sprite_map.h"

//--------------------------------------------------------------------------------------------------------------------//

const size_t kMaximumPixelSpaceValue((1 << 16) - 1);

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteMap::SpriteMap(const TextureHandle& textureHandle, const PixelSpace& frameWidth, const PixelSpace& frameHeight,
	const PixelSpace& spacingX, const PixelSpace& spacingY, const PixelSpace& offsetX, const PixelSpace& offsetY) :
	mTextureHandle(textureHandle),
	mFrameWidth(frameWidth),
	mFrameHeight(frameHeight),
	mSpacingX(spacingX),
	mSpacingY(spacingY),
	mOffsetX(offsetX),
	mOffsetY(offsetY)
{
	tb_error_if(kInvalidTexture == mTextureHandle, "tbExternalError: Expected parameter texture to be a valid texture.");

	const PixelSpace textureWidth(theTextureManager.GetTextureWidth(mTextureHandle));
	const PixelSpace textureHeight(theTextureManager.GetTextureHeight(mTextureHandle));

	tb_error_if(frameWidth < 1 || frameWidth > textureWidth, "tbExternalError: Expected parameter frameWidth to be within range: (1 <= frameWidth <= textureWidth).");
	tb_error_if(frameHeight < 1 || frameHeight > textureHeight, "tbExternalError: Expected parameter frameHeight to be within range: (1 <= frameHeight <= textureHeight).");
	tb_error_if(offsetX > textureWidth - frameWidth, "tbExternalError: Expected parameter offsetX to allow enough room for at least one column of frames.");
	tb_error_if(offsetY > textureHeight - frameHeight, "tbExternalError: Expected parameter offsetY to allow enough room for at least one row of frames.");

	tb_error_if(spacingX > 0, "tbInternalError: Currently SpriteMap does not support spacing between the SpriteFrames.");
	tb_error_if(spacingY > 0, "tbInternalError: Currently SpriteMap does not support spacing between the SpriteFrames.");

	//If the above passed, we should safely be able to get the following information without issue.
	tb_error_if(GetColumnCount() < 1, "tbInternalError: Expected GetColumnCount() to return at least 1.");
	tb_error_if(GetRowCount() < 1, "tbInternalError: Expected GetRowCount() to return at least 1.");
	tb_error_if(GetIndexCount() < 1, "tbInternalError: Expected GetIndexCount() to return at least 1.");
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteMap::~SpriteMap(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteFrame tbGraphics::SpriteMap::GetSpriteFrameAtIndex(const size_t& frameIndex) const
{
	tb_error_if(kInvalidTexture == mTextureHandle, "tbInternalError: Expected parameter texture to be a valid texture at this point.");

	const size_t columnCount(GetColumnCount());
	const size_t frameRow(frameIndex / columnCount);
	const size_t frameColumn(frameIndex - (columnCount * frameRow));
	return GetSpriteFrameAtLocation(frameColumn, frameRow);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteFrame tbGraphics::SpriteMap::GetSpriteFrameAtLocation(const size_t& frameColumn, const size_t& frameRow) const
{
	tb_error_if(kInvalidTexture == mTextureHandle, "tbInternalError: Expected parameter texture to be a valid texture at this point.");

	const size_t frameXFull((mFrameWidth * frameColumn) + mOffsetX);
	tb_error_if(frameXFull >= kMaximumPixelSpaceValue, "tbInternalError: Expected frameX value to fall within PixelSpace range.");

	const size_t frameYFull((mFrameHeight * frameRow) + mOffsetY);
	tb_error_if(frameYFull >= kMaximumPixelSpaceValue, "tbInternalError: Expected frameY value to fall within PixelSpace range.");

	const PixelSpace frameX(static_cast<PixelSpace>(frameXFull));
	const PixelSpace frameY(static_cast<PixelSpace>(frameYFull));
	return SpriteFrame::CreateWith(mTextureHandle, frameX, frameY, mFrameWidth, mFrameHeight);
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbGraphics::SpriteMap::GetIndexCount(void) const
{
	return GetColumnCount() * GetRowCount();
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbGraphics::SpriteMap::GetColumnCount(void) const
{
	tb_error_if(kInvalidTexture == mTextureHandle, "tbInternalError: Expected parameter texture to be a valid texture at this point.");

	const PixelSpace textureWidth(theTextureManager.GetTextureWidth(mTextureHandle));
	return ((textureWidth - mOffsetX) / mFrameWidth);
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbGraphics::SpriteMap::GetRowCount(void) const
{
	tb_error_if(kInvalidTexture == mTextureHandle, "tbInternalError: Expected parameter texture to be a valid texture at this point.");

	const PixelSpace textureHeight(theTextureManager.GetTextureHeight(mTextureHandle));
	return ((textureHeight - mOffsetY) / mFrameHeight);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteMap::GetFrameWidth(void) const
{
	return mFrameWidth;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteMap::GetFrameHeight(void) const
{
	return mFrameHeight;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::SpriteMap::GetTextureHandle(void) const
{
	return mTextureHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::SpriteMap::RenderSpritesByIndex(const tbCore::uint16* const spriteArray, const size_t& columnCount, const size_t& rowCount,
	const tbMath::Vector2& position) const
{
	const size_t kMaximumIndex(GetIndexCount());
	for (size_t row(0); row < rowCount; ++row)
	{
		const size_t rowIndex(row * columnCount);
		const float rowPosition(position.y + (row * mFrameHeight));
		for (size_t column(0); column < columnCount; ++column)
		{
			const tbCore::uint16& spriteFrameIndex(spriteArray[column + rowIndex]);
			if (spriteFrameIndex >= kMaximumIndex)
			{
				continue;
			}

			Sprite temporarySlowMethod(GetSpriteFrameAtIndex(spriteFrameIndex));
			temporarySlowMethod.SetPosition(position.x + (column * mFrameWidth), rowPosition);
			temporarySlowMethod.Render();
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//
