///
/// @file
/// @details This is the start of the texture manager that will eventually become the TurtleBrains texture manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// Credits: Implementation makes heavy use of stb_image.h (v1.33) by Sean Barrett which is in Public Domain
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_texture_manager.h"

#include "../core/tb_opengl.h"
#include "implementation/stb_image.h"
#include "implementation/tbi_renderer.h"
#include "implementation/tbi_blank_texture.h"

#include <climits>
#include <map>
#include <list>
#include <cassert>

const tbGraphics::TextureHandle tbGraphics::kInvalidTexture(0); //0 is not supposed to be a valid texture in gl.

namespace tbImplementation
{
	tbGraphics::TextureHandle tbiBlankTextureHandle(tbGraphics::kInvalidTexture);
};

const tbGraphics::TextureHandle& tbGraphics::kBlankTexture(tbImplementation::tbiBlankTextureHandle);
tbGraphics::TextureManager tbGraphics::theTextureManager;

namespace tbImplementation
{
	struct TextureData
	{
		tbCore::tbString mFilename;
		unsigned int texture;
		tbGraphics::PixelSpace width;
		tbGraphics::PixelSpace height;
		size_t mReferences;

		TextureData(const tbGraphics::PixelSpace& textureWidth, const tbGraphics::PixelSpace& textureHeight, const tbCore::tbString& filename = tb_string(""));
	};

	typedef std::map<tbCore::tbString, tbGraphics::TextureHandle> TextureCacheContainer;
	typedef std::map<tbGraphics::TextureHandle, TextureData> TextureDataContainer;

	tbGraphics::TextureHandle theBoundTexture(tbGraphics::kInvalidTexture);
	TextureCacheContainer theTextureCache;	//Stores texture handles to all textures loaded by file.
	TextureDataContainer theTextures;				//Stores ALL textures, this is where the texture data is.

	const tbGraphics::TextureHandle& GetTextureFromCache(const tbCore::tbString& filename);
	const TextureData& GetTextureData(const tbGraphics::TextureHandle& textureHandle); //Will trigger error condition if handle invalid.
	void IncrementReferenceOf(const tbGraphics::TextureHandle& textureHandle);
	const tbGraphics::TextureHandle& OnCreateTexture(const unsigned char* imageData, TextureData& textureData);
	void OnDestroyTexture(const tbGraphics::TextureHandle& textureHandle);

}; /* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::TextureData::TextureData(const tbGraphics::PixelSpace& textureWidth, const tbGraphics::PixelSpace& textureHeight,
	const tbCore::tbString& filename) :
	mFilename(filename),
	texture(tbGraphics::kInvalidTexture),
	width(textureWidth),
	height(textureHeight),
	mReferences(1)
{
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGraphics::TextureHandle& tbImplementation::GetTextureFromCache(const tbCore::tbString& filename)
{
	TextureCacheContainer::const_iterator textureIterator(theTextureCache.find(filename));
	if (textureIterator != theTextureCache.end())
	{
		return textureIterator->second;
	}
	return tbGraphics::kInvalidTexture;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbImplementation::TextureData& tbImplementation::GetTextureData(const tbGraphics::TextureHandle& textureHandle)
{
	TextureDataContainer::const_iterator textureIterator(theTextures.find(textureHandle));
	tb_error_if(textureIterator == theTextures.end(), "tbError: Attempting to get a texture that does not exist! Invalid Handle.");
	return textureIterator->second;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::IncrementReferenceOf(const tbGraphics::TextureHandle& textureHandle)
{
	TextureDataContainer::iterator textureIterator(theTextures.find(textureHandle));
	tb_error_if(textureIterator == theTextures.end(), "tbError: Attempting to IncrementReferenceOf a handle that was not found.");
	++textureIterator->second.mReferences;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGraphics::TextureHandle& tbImplementation::OnCreateTexture(const unsigned char* imageData, tbImplementation::TextureData& textureData)
{
	//Massive ugly hack.
	static bool initialized = false;
	if (false == initialized)
	{
		initialized = true;

		tbiBlankTextureHandle = tbGraphics::theTextureManager.CreateTextureFromFileData(theTurtleBrainsBlankTextureSourceData, theTurtleBrainsBlankTextureSourceSize);
	}

	tb_error_if(nullptr == imageData, "tbInternalError: Invalid imageData passed to OnCreateTexture()");

	{	//Do a little bit of OpenGL processing for the image data, and make the textureData.texture handle valid.

		tb_check_recent_gl_errors("prior to OnCreateTexture()");

		tb_check_gl_errors(glGenTextures(1, &textureData.texture));
		tb_error_if(tbGraphics::kInvalidTexture == textureData.texture, "tbInternalError: Failed assumption that no gl texture names can be 0.");
		tb_check_gl_errors(glBindTexture(GL_TEXTURE_2D, textureData.texture));

		//Technically we don't want this seen, so we use bright pink.
		float borderColor[] = { 1.0f, 0.0f, 1.0f, 1.0f };
		tb_check_gl_errors(glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, borderColor));
		tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
		tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
		tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT));
		tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT));

		//	It may be best to create the mipmaps, for 3D projects anyway, however glGenerateMipmap not found... (Possibly before glew)
		//	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST));
		//	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_NEAREST));
		//	tb_check_gl_errors(glGenerateMipmap(GL_TEXTURE_2D));

		tb_check_gl_errors(glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, textureData.width, textureData.height, 0, GL_RGBA, GL_UNSIGNED_BYTE, imageData));
	}

	{	//Wrap up creating the texture by storing the data and caching it with filename if possible to avoid multiple loads.
		const tbGraphics::TextureHandle textureHandle(textureData.texture);

		TextureDataContainer::iterator newTextureIterator(theTextures.insert(TextureDataContainer::value_type(textureHandle, textureData)).first);

		if (false == textureData.mFilename.empty())
		{
			theTextureCache.insert(TextureCacheContainer::value_type(textureData.mFilename, textureHandle));
			theTextureCache[textureData.mFilename] = textureHandle;
		}

		tb_error_if(newTextureIterator->first != textureHandle, "tbInternalError: Expected inserted value to match handle.");
		return newTextureIterator->first;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OnDestroyTexture(const tbGraphics::TextureHandle& textureHandle)
{
	if (false == tbImplementation::Renderer::tbiIsRendererAvailable)
	{
		//tb_log("tbInternalWarning: TODO: TIM: Skipping the unloading of a texture, need refactor.");
		return;
	}

	TextureDataContainer::iterator textureDataIterator(theTextures.find(textureHandle));
	if (textureDataIterator != theTextures.end())
	{
		TextureData& textureData(textureDataIterator->second);
		tb_error_if(0 == textureData.mReferences, "tbInternalError: Never ")
		--textureData.mReferences;

		if (0 == textureData.mReferences)
		{
			tb_check_gl_errors(glDeleteTextures(1, &textureData.texture));

			theTextureCache.erase(textureData.mFilename);
			theTextures.erase(textureDataIterator);

			//	This was replaced with the "theTextureCache.erase()" line above, keeping around for a bit.
			//
			//	TextureCacheContainer::iterator textureCacheIterator(theTextureCache.find(textureData.mFilename));
			//	if (textureCacheIterator != theTextureCache.end())
			//	{
			//		theTextureCache.erase(textureCacheIterator);
			//	}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureManager::TextureManager(void) :
	mBoundTexture(kInvalidTexture)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureManager::~TextureManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::TextureManager::CreateTextureFromFile(const tbCore::tbString& filename)
{
	TextureHandle textureHandle = tbImplementation::GetTextureFromCache(filename);
	if (kInvalidTexture != textureHandle)
	{
		tbImplementation::IncrementReferenceOf(textureHandle);
	}
	else //The texture needs to be loaded for the first time.
	{
		int imageWidth(0);
		int imageHeight(0);
		int imageComponents(0);
		stbi_uc* imageData = stbi_load(tbCore::ToStdString(filename).c_str(), &imageWidth, &imageHeight, &imageComponents, 4);

		tb_error_if(nullptr == imageData, "tbExternalError: Texture \"%s\" could not be loaded from file.", filename.c_str());
		tb_error_if(imageWidth < 0 || imageWidth > 0xFFFF, "tbError: Expected imageWidth to be in range: (0 < imageWidth <= 65536), inprecise cast to PixelSpace.");
		tb_error_if(imageHeight < 0 || imageHeight > 0xFFFF, "tbError: Expected imageHeight to be in range: (0 < imageHeight <= 65536), inprecise cast to PixelSpace.");

		tbImplementation::TextureData textureData(static_cast<PixelSpace>(imageWidth), static_cast<PixelSpace>(imageHeight), filename);
		textureHandle = tbImplementation::OnCreateTexture(imageData, textureData);
		stbi_image_free(imageData);
	}
	return textureHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::TextureManager::CreateTextureFromFileData(const unsigned char* fileDataInMemory, const size_t& fileSizeInBytes)
{
	tb_error_if(nullptr == fileDataInMemory, "tbExternalError: The fileDataInMemory parameter is expected to be a valid pointer.");
	tb_error_if(fileSizeInBytes > INT_MAX, "tbError: TextureData fileSizeInBytes too large for following cast into into int, corrupt size?");

	int imageWidth(0);
	int imageHeight(0);
	int imageComponents(0);
	stbi_uc* imageData = stbi_load_from_memory(fileDataInMemory, static_cast<int>(fileSizeInBytes), &imageWidth, &imageHeight, &imageComponents, 4);

	tb_error_if(nullptr == imageData, "tbExternalError: Texture \"%s\" could not be loaded from file data provided, corrupt?");
	tb_error_if(imageWidth < 0 || imageWidth > 0xFFFF, "tbError: Expected imageWidth to be in range: (0 < imageWidth <= 65536), inprecise cast to PixelSpace.");
	tb_error_if(imageHeight < 0 || imageHeight > 0xFFFF, "tbError: Expected imageHeight to be in range: (0 < imageHeight <= 65536), inprecise cast to PixelSpace.");

	tbImplementation::TextureData textureData(static_cast<PixelSpace>(imageWidth), static_cast<PixelSpace>(imageHeight));
	TextureHandle textureHandle = tbImplementation::OnCreateTexture(imageData, textureData);
	stbi_image_free(imageData);
	return textureHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::TextureManager::CreateTextureFromPixelData(const unsigned char* pixelData,
	const PixelSpace& textureWidth, const PixelSpace& textureHeight)
{
	tb_error_if(nullptr == pixelData, "tbExternalError: The pixelData parameter is expected to be a valid pointer.");
	tb_error_if(0 == textureWidth, "tbExternalError: The textureWidth must be greater than 0.");
	tb_error_if(0 == textureHeight, "tbExternalError: The textureHeight must be greater than 0.");

	tbImplementation::TextureData textureData(textureWidth, textureHeight);
	return tbImplementation::OnCreateTexture(pixelData, textureData);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TextureHandle tbGraphics::TextureManager::CreateTextureReference(const TextureHandle& textureHandle)
{
	tb_error_if(kInvalidTexture == textureHandle, "tbExternalError: The textureHandle must be a valid texture handle.");
	tbImplementation::IncrementReferenceOf(textureHandle);
	return textureHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::TextureManager::DestroyTexture(const tbGraphics::TextureHandle& textureHandle)
{
	if (textureHandle == mBoundTexture)
	{
		BindTexture(kInvalidTexture);
	}

	tbImplementation::OnDestroyTexture(textureHandle);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::TextureManager::BindTexture(const tbGraphics::TextureHandle& textureHandle)
{
	mBoundTexture = textureHandle;

	if (kInvalidTexture == mBoundTexture)
	{	//Unbind any textures
		tb_check_gl_errors(glBindTexture(GL_TEXTURE_2D, 0));
	}
	else
	{
		tb_check_gl_errors(glBindTexture(GL_TEXTURE_2D, tbImplementation::GetTextureData(textureHandle).texture));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::TextureManager::GetTextureWidth(const TextureHandle& textureHandle) const
{
	return (kInvalidTexture == textureHandle) ? 0 : tbImplementation::GetTextureData(textureHandle).width;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::TextureManager::GetTextureHeight(const TextureHandle& textureHandle) const
{
	return (kInvalidTexture == textureHandle) ? 0 : tbImplementation::GetTextureData(textureHandle).height;
}

//--------------------------------------------------------------------------------------------------------------------//
