///
/// @file
/// @details An capture/exporter from game to GIF.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tbu_gif_exporter.h"

#include "../../debug_tool_set/tb_debug_logger.h"
#include "../../core/tb_types.h"
#include "../../core/tb_opengl.h"

#include <list>
#include <thread>
#include <ctime>

#ifdef tb_linux
  #define gif_none
#else
  //#define gif_none
  #define gif_multithread
#endif /*tb_platform_check */

#include "implementation/gif.h"

#ifndef gif_none

namespace Gif
{
	GifWriter tbiGifWriter;

#ifdef gif_multithread
	std::thread* tbiGifThread;
	bool tbiThreadRunning(false);
	void ThreadedProcessing(void);
#endif /* gif_multithread */

	const bool kDelayProcessing(true);

	typedef std::list<uint8_t*> FrameContainer;
	FrameContainer tbiFrameData;

	float tbiGifCaptureTimer(-1.0f); //Time remaining before capturing next frame.
	const int kFramesPerSecond(15);
	const float kTimeBetweenFrames(1.0f / static_cast<float>(kFramesPerSecond));

	const size_t kFrameBufferWidth(1280);
	const size_t kFrameBufferHeight(720);
	const size_t kOutputBufferWidth(1280 / 2);
	const size_t kOutputBufferHeight(720 / 2);

	void ProcessFrame(const uint8_t* frameBufferData);
};

#ifdef gif_multithread

void Gif::ThreadedProcessing(void)
{
	tbiThreadRunning = true;

	for (Gif::FrameContainer::iterator itr = Gif::tbiFrameData.begin(); itr != Gif::tbiFrameData.end(); ++itr)
	{
		Gif::ProcessFrame(*itr);
		delete[] * itr;
	}

	Gif::tbiFrameData.clear();

	GifEnd(&Gif::tbiGifWriter);

	tbiThreadRunning = false;
}

#endif /* gif_multithread */

using namespace Gif;

void Gif::ProcessFrame(const uint8_t* frameBufferData)
{
	static uint8_t* outputBufferData = new uint8_t[kOutputBufferWidth * kOutputBufferHeight * 4];

	if (Gif::kOutputBufferWidth == Gif::kFrameBufferWidth)
	{
		for (size_t row(0); row < kFrameBufferHeight; ++row)
		{
			memcpy(&outputBufferData[(kFrameBufferWidth * 4) * ((kFrameBufferHeight - (row + 1)))], &frameBufferData[(kFrameBufferWidth * 4) * row], kFrameBufferWidth * 4);
		}
	}
	else if (Gif::kOutputBufferWidth == Gif::kFrameBufferWidth / 2)
	{
		for (size_t y(0); y < kOutputBufferHeight; ++y)
		{
			for (size_t x(0); x < kOutputBufferWidth; ++x)
			{
				const size_t outputIndex((x * 4) + (kOutputBufferWidth * 4) * ((kOutputBufferHeight - (y + 1))));
				const size_t frameIndex((x * 4 * 2) + (kFrameBufferWidth * 4) * (y * 2));

				for (size_t channel(0); channel < 4; ++channel)
				{
					outputBufferData[outputIndex + channel] = frameBufferData[frameIndex + channel];
				}
			}
		}
	}

	GifWriteFrame(&Gif::tbiGifWriter, outputBufferData, kOutputBufferWidth, kOutputBufferHeight, static_cast<uint32_t>(Gif::kTimeBetweenFrames * 100.0f), 8, false);
}

#endif /* gif_none */

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifPerformCapture(const float deltaTime, const bool toggleCaptureState)
{
	static bool gifCapturing(false);
	if (true == toggleCaptureState)
	{
		gifCapturing = !gifCapturing;
		if (true == gifCapturing)
		{
			TurtleBrains::Graphics::Unstable::GifStartCapture();
		}
		else
		{
			TurtleBrains::Graphics::Unstable::GifFinalCapture();
		}
	}

	if (true == gifCapturing)
	{
		TurtleBrains::Graphics::Unstable::GifUpdateCapture(deltaTime);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifCaptureCleanup(void)
{
	GifJoinThread();
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifStartCapture(void)
{
#ifndef gif_none

#ifdef gif_multithread
	if (true == Gif::tbiThreadRunning)
	{
		Gif::tbiGifThread->join();
	}
#endif /* gif_multithread */

	char filename[64];
	std::time_t time = std::time(nullptr);
#if defined(tb_visual_cpp)
	std::tm localTime;
	localtime_s(&localTime, &time);
#else
	std::tm localTime = *std::localtime(&time);
#endif
	strftime(filename, 64, "%Y%m%d_%H%M%S_screencap.gif", &localTime);

	GifBegin(&Gif::tbiGifWriter, filename, kOutputBufferWidth, kOutputBufferHeight, static_cast<uint32_t>(Gif::kTimeBetweenFrames * 100.0f), 8, false);

	tbiGifCaptureTimer = 0.0f;

	tb_log("Gif Capture Started, saving to \"%s\"", filename);

#endif /* gif_none */
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifCaptureFrame(void)
{
#ifndef gif_none

	uint8_t* frameBufferData = new uint8_t[kFrameBufferWidth * kFrameBufferHeight * 4];
	glReadPixels(0, 0, kFrameBufferWidth, kFrameBufferHeight, GL_RGBA, GL_UNSIGNED_BYTE, frameBufferData);

	if (true == kDelayProcessing)
	{
		Gif::tbiFrameData.push_back(frameBufferData);
	}
	else
	{
		Gif::ProcessFrame(frameBufferData);
		delete [] frameBufferData;
	}

#endif /* gif_none */
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifUpdateCapture(const float deltaTime)
{
#ifndef gif_none

	tbiGifCaptureTimer -= deltaTime;
	if (tbiGifCaptureTimer < 0.0f)
	{
		GifCaptureFrame();
		tbiGifCaptureTimer += Gif::kTimeBetweenFrames;
	}

#endif /* gif_none */
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifFinalCapture(void)
{
#ifndef gif_none

#ifdef gif_multithread
	//Gif::tbiGifThread = std::thread(Gif::ThreadedProcessing);
	Gif::tbiGifThread = new std::thread(Gif::ThreadedProcessing);
#else

	if (true == kDelayProcessing)
	{
		for (Gif::FrameContainer::iterator itr = Gif::tbiFrameData.begin(); itr != Gif::tbiFrameData.end(); ++itr)
		{
			Gif::ProcessFrame(*itr);
			delete [] *itr;
		}

		Gif::tbiFrameData.clear();

		GifEnd(&Gif::tbiGifWriter);
	}
#endif

	tb_log("Gif Capture Complete, saved.");

#endif /* gif_none */
}

//--------------------------------------------------------------------------------------------------------------------//

void TurtleBrains::Graphics::Unstable::GifJoinThread(void)
{
#ifndef gif_none
	Gif::tbiGifThread->join();
	delete Gif::tbiGifThread;
	Gif::tbiGifThread = nullptr;
#endif /* gif_none */
}

//--------------------------------------------------------------------------------------------------------------------//
