///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Interpolation_h_
#define _TurtleBrains_Interpolation_h_

namespace TurtleBrains
{
	namespace Math
	{
		namespace Interpolation
		{

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f and returns the percentage value as if
			///   it had been interpolated linearly, which is the same as returning the same value.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f to be linearly interpolated and returned
			///   as a value still within the range 0.0f to 1.0f.
			///
			inline float Linear(float percentage)
			{
				return percentage;
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f linearly interpolates between the start
			///   and final value returning the value linearly interpolated based on the percentage with 0.0f being
			///   the start value and 1.0f being the final.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value and 1.0f being the final value.
			/// @param start A value representing the starting point of the interpolation when percentage is at 0.0f.
			/// @param final A value representing the finishing point of the interpolation when percentage is at 1.0f.
			///
			/// @note This is a templated function that will work on any type supplied by start/final as long as the
			///   operators: +, - of the Type and * by float are overloaded.
			///
			template <typename T> T Linear(float percentage, const T& start, const T& final)
			{
				//return (p * a) + (b * (1 - p));
				return start + ((final - start) * percentage);
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f and returns the percentage value as if
			///   it had been interpolated with a smooth step, which slowly increases rate of change then slowly comes
			///   to the final value.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f to be interpolated with smooth step and 
			///   returned as a value still within the range 0.0f to 1.0f.
			///
			inline float SmoothStep(float percentage)
			{
				return (percentage * percentage * (3.0f - 2.0f * percentage));
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f interpolates with a smooth step between
			///   the start and final value, returning a value that is interpolated with the smooth stepped percentage,
			///   with 0.0f being the start value and 1.0f being the final.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value and 1.0f being the final value.
			/// @param start A value representing the starting point of the interpolation when percentage is at 0.0f.
			/// @param final A value representing the finishing point of the interpolation when percentage is at 1.0f.
			///
			/// @note This is a templated function that will work on any type supplied by start/final as long as the
			///   operators: + by Type and * by float are overloaded.
			///
			template <typename T> T SmoothStep(float percentage, const T& start, const T& end)
			{
				const float smoothPercentage = SmoothStep(percentage);
				return (start * smoothPercentage) + (end * (1.0f - smoothPercentage));
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f and returns the percentage value as if
			///   it had been interpolated with a square, which quickly increase the distance covered then comes to a
			///   slow stop.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f to be interpolated with squared pattern and 
			///   returned as a value still within the range 0.0f to 1.0f.
			///
			inline float Squared(float percentage)
			{
				return percentage * percentage;
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f interpolates with an squared between
			///   the start and final value, returning a value that is interpolated with the squared interpolated
			///   percentage, with 0.0f being the start value and 1.0f being the final.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value and 1.0f being the final value.
			/// @param start A value representing the starting point of the interpolation when percentage is at 0.0f.
			/// @param final A value representing the finishing point of the interpolation when percentage is at 1.0f.
			///
			/// @note This is a templated function that will work on any type supplied by start/final as long as the
			///   operators: + by Type and * by float are overloaded.
			///
			template <typename T> T Squared(float percentage, const T& start, const T& end)
			{
				const float squaredPercentage = Squared(percentage);
				return (squaredPercentage * start) + (end * (1.0f - squaredPercentage));
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f and returns the percentage value as if
			///   it had been interpolated with the inversed square, which slowly increase the distance covered then 
			///	  gets faster as it gets nearer to 1.0f.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f to be interpolated with inverse squared 
			///   pattern and  returned as a value still within the range 0.0f to 1.0f.
			///
			inline float InverseSquared(float percentage)
			{
				return 1.0f - ((1.0f - percentage) * (1.0f - percentage));
			}

			///
			/// @details Given a percentage value in the range of 0.0f to 1.0f interpolates with an inverse square between
			///   the start and final value, returning a value that is interpolated with the inverse square interpolated
			///   percentage, with 0.0f being the start value and 1.0f being the final.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value and 1.0f being the final value.
			/// @param start A value representing the starting point of the interpolation when percentage is at 0.0f.
			/// @param final A value representing the finishing point of the interpolation when percentage is at 1.0f.
			///
			/// @note This is a templated function that will work on any type supplied by start/final as long as the
			///   operators: + by Type and * by float are overloaded.
			///
			template <typename T> T InverseSquared(float percentage, const T& start, const T& end)
			{
				const float inverseSquared = InverseSquared(percentage);
				return (inverseSquared * start) + (end * (1.0f - inverseSquared));
			}

			///
			/// @details Using several linear interpolation calls this will interpolate a cubic bezier curve along the
			///   points a, b, c and d.  Where the returned value would start at point a when percentage is 0.0f and
			///   finish at point d when the percentage is 1.0f.  The path will not actually reach point b or c as
			///   these are just control points.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value of point a and 1.0f being the final value of point d.
			/// @param a the start point of the path to follow, this would be returned when percentage is 0.0f.
			/// @param b the first control point of the bezier, initially the movement will start moving from the start
			///   point a to this location but will quickly deviate from this path.
			/// @param c the second control point of the bezier, the final movements will end moving toward the final
			///   point d from this location.
			/// @param d the final point of the path to follow, this would be returned when the percentage is 1.0f.
			///
			/// @note This is a templated function that will work on any type supplied by a,b,d,c as long as the
			///   operators: + and - by Type and * by float are overloaded.
			///
			template <typename T> T CubicBezier(float percentage, const T& a, const T& b, const T& c, const T& d)
			{
				const T tempA = Linear(percentage, a, c);
				const T tempB = Linear(percentage, c, d);
				const T tempC = Linear(percentage, d, b);

				const T tempAA = Linear(percentage, tempA, tempB);
				const T tempBB = Linear(percentage, tempB, tempC);

				return Linear(percentage, tempAA, tempBB);
			}

			///
			/// @details Computes the tangent of the bezier curve at a given point along the curve and returns the
			///   result.
			///
			/// @param percentage A value within the range of 0.0f to 1.0f of the progression, 0.0f being the start
			///   value of point a and 1.0f being the final value of point d.
			/// @param a the start point of the path to follow.
			/// @param b the first control point of the bezier.
			/// @param c the second control point of the bezier.
			/// @param d the final point of the path to follow.
			///
			/// @note This is a templated function that will work on any type supplied by a,b,d,c as long as the
			///   operators: + and - by Type and * by float are overloaded.
			///
			template <typename T> T CubicBezierTangent(float percentage, const T& a, const T& b, const T& c, const T& d)
			{	//Source: http://stackoverflow.com/questions/4089443/find-the-tangent-of-a-point-on-a-cubic-bezier-curve-on-an-iphone

				const T c1(d - (3.0f * c) + (3.0f * b) - a);
				const T c2((3.0f * c) - (6.0f * b) + (3.0f * a));
				const T c3((3.0f * b) - (3.0f * a));
				//const float c4(a);

				return ((c1 * (3.0f * percentage * percentage)) + (c2 * (2.0f * percentage)) + c3);
			}

		}; /* namespace Interpolation */
	}; /* namespace Math */
}; /* namespace TurtleBrains */

namespace tbMath = TurtleBrains::Math;

#endif /* _TurtleBrains_Interpolation_h_ */
