///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Matrix_h_
#define _TurtleBrains_Matrix_h_

#include <cmath>
#include <cstring> //memcpy / memset

#include "../core/tb_configuration.h"
#include "../core/tb_error.h"
#include "tb_vector.h"

namespace TurtleBrains
{
	namespace Math
	{

		///
		/// @details The Matrix3 is a matrix of 9 float components that build a 3x3 matrix for transforms and other
		///   mathematical purposes.
		///
		/// @note The 3x3 Matrix has not been tested as thoroughly as the 4x4 matrix which has been around since about 2005.
		///
		class Matrix3
		{
		public:
			///
			/// @details A constant Matrix3 value representing a zero'd matrix.
			///
			static const Matrix3 kZero;

			///
			/// @details A constant Matrix3 value representing the identity matrix, with zeros for all components
			///   except for the diagonal:
			///     { 1.0f, 0.0f, 0.0f,
			///       0.0f, 1.0f, 0.0f,
			///       0.0f, 0.0f, 1.0f }
			///
			static const Matrix3 kIdentity;

			union
			{
				float mComponents[9];

#if defined(tb_visual_cpp)
#pragma warning(push)
#pragma warning(disable: 4201)
				struct {	//May be deprecating!?
					float m_f11; float m_f21; float m_f31;
					float m_f12; float m_f22; float m_f32;
					float m_f13; float m_f23; float m_f33;
				};
#pragma warning(pop)
#else
				struct {	//May be deprecating!?
					float m_f11; float m_f21; float m_f31;
					float m_f12; float m_f22; float m_f32;
					float m_f13; float m_f23; float m_f33;
				};
#endif
			};

			///
			/// @details Constructs a Matrix3 object with the supplied components.
			///
			inline explicit Matrix3(	float f11 = 0.0f, float f21 = 0.0f, float f31 = 0.0f,
										float f12 = 0.0f, float f22 = 0.0f, float f32 = 0.0f,
										float f13 = 0.0f, float f23 = 0.0f, float f33 = 0.0f)
			{
				//m_f11 = f11; m_f21 = f21; m_f31 = f31;
				//m_f12 = f12; m_f22 = f22; m_f32 = f32;
				//m_f13 = f13; m_f23 = f23; m_f33 = f33;

				mComponents[0] = f11; mComponents[1] = f21; mComponents[2] = f31;
				mComponents[3] = f12; mComponents[4] = f22; mComponents[5] = f32;
				mComponents[6] = f13; mComponents[7] = f23; mComponents[8] = f33;
			}

			///
			/// @details Constructs a Matrix3 object with the supplied float array.
			///
			/// @param componentArray Must have at least 9 floats, one for each component in the Matrix3.
			///
			inline explicit Matrix3(const float * const componentArray)
			{
				memcpy(mComponents, componentArray, sizeof(float) * 9);
			}

			///
			/// @details Constructs a Matrix3 object leaving the members uninitialized and must be initialized manually before
			///   the object is used.
			///
			/// @note Should be used only if the consequences are understood, and even then only sparingly.
			///
			inline explicit Matrix3(const SkipInitialization& fastAndStupid)
			{
				tb_unused(fastAndStupid);
			}

			///
			/// @details Constructs a Matrix4 object by copying the component values from an existing Matrix3 object.
			///
			inline Matrix3(const Matrix3& other)
			{
				memcpy(mComponents, other.mComponents, sizeof(float) * 9);
			}

			///
			/// @details Assigns each component in the Matrix object to match an existing Matrix3 object.
			///
			inline Matrix3& operator=(const Matrix3& other)
			{
				if (this != &other)
				{
					memcpy(mComponents, other.mComponents, sizeof(float) * 9);
				}
				return (*this);
			}

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the index which can
			///   be from 0 to 8 for a Matrix3 object.
			///
			inline const float& operator[](const size_t& index) const { return mComponents[index]; }

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the index which can
			///   be from 0 to 8 for a Matrix3 object.  The component is returned by reference and can then be modified
			///   externally.  Do so at your own risk.
			///
			inline float& operator[](const size_t& index) { return mComponents[index]; }

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the column and
			///   row parameters.
			///
			inline const float& operator()(const size_t& column, const size_t& row) const { return mComponents[column + (row * 3)]; }

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the column and
			///   row parameters.  The component is returned by reference and can then be modified externally.  Do
			///   so at your own risk.
			///
			inline float& operator()(const size_t& column, const size_t& row) { return mComponents[column + (row * 3)]; }

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the column and
			///   row parameters.
			///
			inline const float& operator()(int column, int row) const { return mComponents[column + (row * 3)]; }

			///
			/// @details This operator returns a specific component of the Matrix3 object picked by the column and
			///   row parameters.  The component is returned by reference and can then be modified externally.  Do
			///   so at your own risk.
			///
			inline float& operator()(int column, int row) { return mComponents[column + (row * 3)]; }

			///
			/// @details This operator returns the array of components used by the Matrix4 object.  If the returned
			///   array is modified, the internal array will also be modified.  Only use this if you understand the
			///   meaning.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline operator float*(void) { return mComponents; }

			///
			/// @details This operator returns the array of components used by the Matrix4 object.
			///
			inline operator const float*(void) const { return mComponents; }

			///
			/// @details Returns a pointer to a Vector3 object whos components are the same components as the basis it
			///   represents.  This is making assumptions that the Vector3 internally operates on an array of three 
			///   floats.  Modifying the returned Vector3 will modify the Matrix3.  Do so at your own risk.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline Vector3* GetBasis(const size_t& basisIndex) { return (Vector3*)&mComponents[basisIndex * 3]; }

			///
			/// @details Returns a pointer to a Vector3 object whos components are the same components as the basis it
			///   represents.  This is making assumptions that the Vector3 internally operates on an array of three 
			///   floats.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline const Vector3* GetBasis(const size_t& basisIndex) const { return (const Vector3*)&mComponents[basisIndex * 3]; }

			///
			/// @details Sets the components of the Matrix3 basis to the values set in the supplied Vector object.
			///
			/// @param basisIndex The basis to set, 0 is the X basis, 1 is the Y basis and 2 is the Z basis.
			/// @param basis The value of each component of the basis vector to be set in the Matrix object.
			///
			inline void SetBasis(const size_t& basisIndex, const Vector3 &basis) { *((Vector3*)&mComponents[basisIndex * 3]) = basis; }

			///
			/// @details Sets the components of the Matrix3 basis to the values supplied by the basisX, Y, Z parameters.
			///
			/// @param basisIndex The basis to set, 0 is the X basis, 1 is the Y basis and 2 is the Z basis.
			/// @param basisX The Value to set the basis X component to.
			/// @param basisY The Value to set the basis Y component to.
			/// @param basisZ The Value to set the basis X component to.
			///
			inline void SetBasis(const size_t& basisIndex, float basisX, float basisY, float basisZ)
			{
				mComponents[(basisIndex * 3) + 0] = basisX;
				mComponents[(basisIndex * 3) + 1] = basisY;
				mComponents[(basisIndex * 3) + 2] = basisZ;
			}
		};

///------------------------------------------------------------------------------------------------------------------///
///------------------------------------------------------------------------------------------------------------------///
///------------------------------------------------------------------------------------------------------------------///

		///
		/// @details The Matrix4 is a matrix of 16 float components that build a 4x4 matrix for transforms and other
		///   mathematical purposes.
		///
		class Matrix4
		{
		public:

			///
			/// @details A constant Matrix4 value representing a zero'd matrix.
			///
			static const Matrix4 kZero;
			
			///
			/// @details A constant Matrix4 value representing the identity matrix, with zeros for all components
			///   except for the diagonal:
			///     { 1.0f, 0.0f, 0.0f, 0.0f,
			///       0.0f, 1.0f, 0.0f, 0.0f,
			///       0.0f, 0.0f, 1.0f, 0.0f,
			///       0.0f, 0.0f, 0.0f, 1.0f }
			///			
			static const Matrix4 kIdentity;

			union
			{
				float mComponents[16];

#if defined(tb_visual_cpp)
#pragma warning(push)
#pragma warning(disable: 4201)
				struct {	//May be deprecating!?
					float m_f11; float m_f21; float m_f31; float m_f41;
					float m_f12; float m_f22; float m_f32; float m_f42;
					float m_f13; float m_f23; float m_f33; float m_f43;
					float m_f14; float m_f24; float m_f34; float m_f44;
				};
#pragma warning(pop)
#else
				struct {	//May be deprecating!?
					float m_f11; float m_f21; float m_f31; float m_f41;
					float m_f12; float m_f22; float m_f32; float m_f42;
					float m_f13; float m_f23; float m_f33; float m_f43;
					float m_f14; float m_f24; float m_f34; float m_f44;
				};
#endif
			};

			///
			/// @details Constructs a Matrix4 object with the supplied components.
			///
			inline explicit Matrix4(	float f11 = 0.0f, float f21 = 0.0f, float f31 = 0.0f, float f41 = 0.0f,
							float f12 = 0.0f, float f22 = 0.0f, float f32 = 0.0f, float f42 = 0.0f,
							float f13 = 0.0f, float f23 = 0.0f, float f33 = 0.0f, float f43 = 0.0f,
							float f14 = 0.0f, float f24 = 0.0f, float f34 = 0.0f, float f44 = 0.0f)
			{
				//m_f11 = f11; m_f21 = f21; m_f31 = f31; m_f41 = f41;
				//m_f12 = f12; m_f22 = f22; m_f32 = f32; m_f42 = f42;
				//m_f13 = f13; m_f23 = f23; m_f33 = f33; m_f43 = f43;
				//m_f14 = f14; m_f24 = f24; m_f34 = f34; m_f44 = f44;

				mComponents[0] = f11; mComponents[1] = f21; mComponents[2] = f31; mComponents[3] = f41;
				mComponents[4] = f12; mComponents[5] = f22; mComponents[6] = f32; mComponents[7] = f42;
				mComponents[8] = f13; mComponents[9] = f23; mComponents[10]= f33; mComponents[11]= f43;
				mComponents[12]= f14; mComponents[13]= f24; mComponents[14]= f34; mComponents[15]= f44;
			}

			///
			/// @details Constructs a Matrix4 object with the supplied float array.
			///
			/// @param componentArray Must have at least 16 floats, one for each component in the Matrix4.
			///
			inline explicit Matrix4(const float * const componentArray)
			{
				memcpy(mComponents, componentArray, sizeof(float) << 4);  //Same as (sizeof(float) * 16)
			}

			///
			/// @details Constructs a Matrix4 object leaving the members uninitialized and must be initialized manually before
			///   the object is used.
			///
			/// @note Should be used only if the consequences are understood, and even then only sparingly.
			///
			inline explicit Matrix4(const SkipInitialization& fastAndStupid)
			{
				tb_unused(fastAndStupid);
			}

			///
			/// @details Constructs a Matrix4 object by copying the component values from an existing Matrix4 object.
			///
			inline Matrix4(const Matrix4& other)
			{
				memcpy(mComponents, other.mComponents, sizeof(float) << 4);  //Same as (sizeof(float) * 16)
			}

			///
			/// @details Assigns each component in the Matrix object to match an existing Matrix4 object.
			///
			inline Matrix4& operator=(const Matrix4& other)
			{
				if (this != &other)
				{
					memcpy(mComponents, other.mComponents, sizeof(float) << 4);  //Same as (sizeof(float) * 16)
				}
				return (*this);
			}

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the index which can
			///   be from 0 to 15 for a Matrix4 object.
			///
			inline const float& operator[](const size_t& index) const { return mComponents[index]; }

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the index which can
			///   be from 0 to 15 for a Matrix4 object.  The component is returned by reference and can then be modified
			///   externally.  Do so at your own risk.
			///
			inline float& operator[](const size_t& index) { return mComponents[index]; }

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the column and
			///   row parameters.
			///
			inline const float& operator()(const size_t& column, const size_t& row) const { return mComponents[column + (row << 2)]; }

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the column and
			///   row parameters.  The component is returned by reference and can then be modified externally.  Do
			///   so at your own risk.
			///
			inline float& operator()(const size_t& column, const size_t& row) { return mComponents[column + (row << 2)]; }

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the column and
			///   row parameters.
			///
			inline const float& operator()(int column, int row) const { return mComponents[column + (row << 2)]; }

			///
			/// @details This operator returns a specific component of the Matrix4 object picked by the column and
			///   row parameters.  The component is returned by reference and can then be modified externally.  Do
			///   so at your own risk.
			///
			inline float& operator()(int column, int row) { return mComponents[column + (row << 2)]; }


			///
			/// @details This operator returns the array of components used by the Matrix4 object.  If the returned
			///   array is modified, the internal array will also be modified.  Only use this if you understand the
			///   meaning.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline operator float*(void) { return mComponents; }

			///
			/// @details This operator returns the array of components used by the Matrix4 object.
			///
			inline operator const float*(void) const { return mComponents; }


			///
			/// @details Returns a pointer to a Vector3 object whos components are the same components as the basis it
			///   represents.  This is making assumptions that the Vector3 internally operates on an array of three 
			///   floats.  Modifying the returned Vector3 will modify the Matrix4.  Do so at your own risk.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline Vector3* GetBasis(const size_t& basisIndex) { return (Vector3*)&mComponents[basisIndex << 2]; }

			///
			/// @details Returns a pointer to a Vector3 object whos components are the same components as the basis it
			///   represents.  This is making assumptions that the Vector3 internally operates on an array of three 
			///   floats.  Modifying the returned Vector3 will modify the Matrix4.  Do so at your own risk.
			///
			/// @note This may actually become deprecated in the future, use at your own risk.
			///
			inline const Vector3* GetBasis(const size_t& basisIndex) const { return (const Vector3*)&mComponents[basisIndex<<2]; }

			///
			/// @details Sets the components of the Matrix4 basis to the values set in the supplied Vector object.
			///
			/// @param basisIndex The basis to set, 0 is the X basis, 1 is the Y basis and 2 is the Z basis.
			/// @param basis The value of each component of the basis vector to be set in the Matrix object.
			///
			inline void SetBasis(const size_t& basisIndex, const Vector3 &basis) { *((Vector3*)&mComponents[basisIndex<<2]) = basis; }

			///
			/// @details Sets the components of the Matrix4 basis to the values supplied by the basisX, Y, Z parameters.
			///
			/// @param basisIndex The basis to set, 0 is the X basis, 1 is the Y basis and 2 is the Z basis.
			/// @param basisX The Value to set the basis X component to.
			/// @param basisY The Value to set the basis Y component to.
			/// @param basisZ The Value to set the basis X component to.
			///
			inline void SetBasis(const size_t& basisIndex, float basisX, float basisY, float basisZ) 
			{ 
				mComponents[(basisIndex<<2) + 0] = basisX;
				mComponents[(basisIndex<<2) + 1] = basisY;
				mComponents[(basisIndex<<2) + 2] = basisZ; 
			}

#ifndef tb_math_less_operators
			///
			/// @details Much like the GetBasis accesses a specific axis of the Matrix4 object, this will return the
			///   amount of translation the transform Matrix4 has.  However it can not be modified and due to the
			///   location of the position components, a temporary Vector3 must be created.
			///
			inline Vector3 GetPosition(void) const	{ return Vector3(m_f14, m_f24, m_f34); }

			///
			/// @details Much like the SetBasis modified a specific axis of the Matrix4 object, this will modify the
			///   translation/position of the transform Matrix4 to be the values of the Vector3 object supplied.
			///
			inline void SetPosition(const Vector3& position)
			{
				m_f14 = position.x;
				m_f24 = position.y;
				m_f34 = position.z;
			}

			///
			/// @details Much like the SetBasis modified a specific axis of the Matrix4 object, this will modify the
			///   translation/position of the transform Matrix4 to be the values supplied.
			///
			inline void SetPosition(float x, float y, float z)	{ m_f14 = x; m_f24 = y; m_f34 = z; }

			///
			/// @details Creates a matrix object with the components transposed and returns the results.
			///
			inline Matrix4 GetTransposed(void) const
			{
				//result->m_f11 = input->m_f11;	result->m_f12 = input->m_f21;	result->m_f13 = input->m_f31;	result->m_f14 = input->m_f41;
				//result->m_f21 = input->m_f12;	result->m_f22 = input->m_f22;	result->m_f23 = input->m_f32;	result->m_f24 = input->m_f42;
				//result->m_f31 = input->m_f13;	result->m_f32 = input->m_f23;	result->m_f33 = input->m_f33;	result->m_f34 = input->m_f43;
				//result->m_f41 = input->m_f14;	result->m_f42 = input->m_f24;	result->m_f43 = input->m_f34;	result->m_f44 = input->m_f44;

				Matrix4 temp;
				const Matrix4& self(*this);
				temp(0, 0) = self(0, 0);    temp(0, 1) = self(1, 0);    temp(0, 2) = self(2, 0);    temp(0, 3) = self(3, 0);
				temp(1, 0) = self(0, 1);    temp(1, 1) = self(1, 1);    temp(1, 2) = self(2, 1);    temp(1, 3) = self(3, 1);
				temp(2, 0) = self(0, 2);    temp(2, 1) = self(1, 2);    temp(2, 2) = self(2, 2);    temp(2, 3) = self(3, 2);
				temp(3, 0) = self(0, 3);    temp(3, 1) = self(1, 3);    temp(3, 2) = self(2, 3);    temp(3, 3) = self(3, 3);
				return temp;
			}
#endif
		};

///------------------------------------------------------------------------------------------------------------------///
///------------------------------------------------------------------------------------------------------------------///
///------------------------------------------------------------------------------------------------------------------///

		///
		/// @details Fills the matrix with the values from the identity matrix and returns the result.
		///
		/// @param result The matrix to fill out and return.
		///
		inline Matrix4* MatrixCreateIdentity(Matrix4* result)
		{
			//memset(&matR->m_f11, 0, sizeof(float) << 4);
			//matR->m_f11 = 1.0f;
			//matR->m_f22 = 1.0f;
			//matR->m_f33 = 1.0f;
			//matR->m_f44 = 1.0f;
			//return matR;

			*result = Matrix4::kIdentity;
			return result;
		}
		
		///
		/// @details Creates a transform Matrix from a forward vector and up vector, similar to creating a LookAt.
		///
		/// @param result The matrix to fill out with the resulting LookAt matrix.
		/// @param forward The direction the transform matrix will be pointing at, must be a unit vector of the direction.
		/// @param up An optional unit vector representing the up vector, if left NULL (0, 1, 0) will be used as a default up.
		///
		/// @note Currently not yet implemented/tested so this will trigger an error condition if used.
		///
		inline Matrix4* MatrixCreateFromForward(Matrix4* result, const Vector3* forward, const Vector3* up = NULL)
		{
			tb_error(true, "Not yet implemented / tested.");
			MatrixCreateIdentity(result);

			Vector3 realUp((NULL == up) ? Vector3(0.0f, 1.0f, 0.0f) : *up);
			Vector3 right, finalUp;
			Vector3CrossProduct(&right, &realUp, forward);
			Vector3CrossProduct(&finalUp, &right, forward);
			//TODO: TIM: Implementation: This would be useful, implement it for future usage!
			return result;
		}

		///
		/// @details Creates a translation transform Matrix object with no rotation set to the position specified by the Vector.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param translation The amount of translation to add in each axis represented by the vector components.
		///
		inline Matrix4* MatrixCreateTranslation(Matrix4* result, const Vector3* translation)
		{	//Changed September 19, 2010
			MatrixCreateIdentity(result);
			result->m_f14 = translation->x;//matR->m_f41 = vT->x;
			result->m_f24 = translation->y;//matR->m_f42 = vT->y;
			result->m_f34 = translation->z;//matR->m_f43 = vT->z;
			return result;
		}

		///
		/// @details Creates a translation transform Matrix object with no rotation set to the position specified by
		///   the x, y, z parameter values.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param translationX The amount of translation to add to the transform matrix in the X-axis.
		/// @param translationY The amount of translation to add to the transform matrix in the Y-axis.
		/// @param translationZ The amount of translation to add to the transform matrix in the Z-axis.
		///
		inline Matrix4* MatrixCreateTranslation(Matrix4* result, const float translationX, const float translationY, const float translationZ)
		{	//Changed September 19, 2010
			MatrixCreateIdentity(result);
			result->m_f14 = translationX;//matR->m_f41 = fX;
			result->m_f24 = translationY;//matR->m_f42 = fY;
			result->m_f34 = translationZ;//matR->m_f43 = fZ;
			return result;
		}

		///
		/// @details Creates a scale transform Matrix object with no translation or rotation set to the position 
		///   specified by the Vector object.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param scale The amount of scaling of each axis represented by the vector components.
		///
		inline Matrix4* MatrixCreateScale(Matrix4* result, const Vector3* scale)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);
			result->m_f11 = scale->x;
			result->m_f22 = scale->y;
			result->m_f33 = scale->z;
			result->m_f44 = 1.0f;
			return result;
		}

		///
		/// @details Creates a scale transform Matrix object with no translation or rotation set to the position 
		///   specified by the x, y, z parameter values.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param scaleX The amount of scaling for the transform matrix in the X-axis.
		/// @param scaleY The amount of scaling for the transform matrix in the Y-axis.
		/// @param scaleZ The amount of scaling for the transform matrix in the Z-axis.
		///
		inline Matrix4* MatrixCreateScale(Matrix4* result, const float scaleX, const float scaleY, const float scaleZ)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);
			result->m_f11 = scaleX;
			result->m_f22 = scaleY;
			result->m_f33 = scaleZ;
			result->m_f44 = 1.0f;
			return result;
		}
		
		///
		/// @details Creates a rotation transform Matrix object with no translation or scaling that will rotate around
		///   the X-axis by the amount specified by the rotation parameter.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param rotationInDegrees The amount to rotate around the X-axis in degrees.
		///
		inline Matrix4* MatrixCreateRotationX(Matrix4* result, const float rotationInDegrees)
		{
			const float rotationInRadians(tbMath::Convert::DegreesToRadians(rotationInDegrees));

			MatrixCreateIdentity(result);
			result->m_f22 = cos(rotationInRadians);
			result->m_f32 = -sin(rotationInRadians);
			result->m_f23 = sin(rotationInRadians);
			result->m_f33 = cos(rotationInRadians);
			return result;
		}

		///
		/// @details Creates a rotation transform Matrix object with no translation or scaling that will rotate around
		///   the Y axis by the amount specified by the rotation parameter.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param rotationInDegrees The amount to rotate around the Y-axis in degrees.
		///
		inline Matrix4* MatrixCreateRotationY(Matrix4* result, const float rotationInDegrees)
		{
			const float rotationInRadians(tbMath::Convert::DegreesToRadians(rotationInDegrees));

			MatrixCreateIdentity(result);
			result->m_f11 = cos(rotationInRadians);
			result->m_f31 = sin(rotationInRadians);
			result->m_f13 = -sin(rotationInRadians);
			result->m_f33 = cos(rotationInRadians);
			return result;
		}

		///
		/// @details Creates a rotation transform Matrix object with no translation or scaling that will rotate around
		///   the X-axis by the amount specified by the rotation parameter.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param rotationInDegrees The amount to rotate around the Z-axis in degrees.
		///
		inline Matrix4* MatrixCreateRotationZ(Matrix4* result, const float rotationInDegrees)
		{
			const float rotationInRadians(tbMath::Convert::DegreesToRadians(rotationInDegrees));

			MatrixCreateIdentity(result);
			result->m_f11 = cos(rotationInRadians);
			result->m_f21 = -sin(rotationInRadians);
			result->m_f12 = sin(rotationInRadians);
			result->m_f22 = cos(rotationInRadians);
			return result;
		}

		///
		/// @details Creates a rotation transform Matrix object with no translation or scaling that will rotate around
		///   the specied rotationAxis by the amount specified by the rotation parameter.
		///
		/// @param result The matrix to hold the result and be returned.
		/// @param rotationAxis The axis to rotate around, should be a unit vector, can be an arbitrary direction.
		/// @param rotationInDegrees The amount to rotate around the Z-axis in degrees.
		///
		inline Matrix4* MatrixCreateRotationA(Matrix4* result, const Vector3* rotationAxis, const float rotationInDegrees)
		{
			const float rotationInRadians(tbMath::Convert::DegreesToRadians(rotationInDegrees));

			// Rotations.
			const float s = sinf(rotationInRadians);
			const float c = cosf(rotationInRadians);
			const float t = 1.0f - c;

			float x = rotationAxis->x;
			float y = rotationAxis->y;
			float z = rotationAxis->z;

			const float tx = t * rotationAxis->x;
			const float ty = t * rotationAxis->y;
			const float tz = t * rotationAxis->z;
			const float sx = s * rotationAxis->x;
			const float sy = s * rotationAxis->y;
			const float sz = s * rotationAxis->z;

			result->mComponents[0] = tx * x + c;
			result->mComponents[1] = tx * y + sz;
			result->mComponents[2] = tx * z - sy;
			result->mComponents[3] = 0.0f;

			result->mComponents[4]  = tx * y - sz;
			result->mComponents[5]  = ty * y + c;
			result->mComponents[6]  = ty * z + sx;
			result->mComponents[7]  = 0.0f;

			result->mComponents[8]  = tx * z + sy;
			result->mComponents[9]  = ty * z - sx;
			result->mComponents[10] = tz * z + c;
			result->mComponents[11] = 0.0f;

			result->mComponents[12] = 0.0f;
			result->mComponents[13] = 0.0f; 
			result->mComponents[14] = 0.0f;
			result->mComponents[15] = 1.0f; 

			return result;
		}

		///
		/// @details Creates a right-handed projection matrix for 3D rendering given a few setup parameters.
		///
		/// @param result Where to store the resulting matrix and return.
		/// @param fieldOfView The vertical field of vision for the projection matrix.
		/// @param aspectRatio The aspect ratio of the window/screen or target render.
		/// @param nearPlane The distance to the near plane of the projection matrix.
		/// @param farPlane The distance to the far plane of the projection matrix.
		///
		inline Matrix4* MatrixCreatePerspectiveRH(Matrix4* result, const float fieldOfView, const float aspectRatio, 
			const float nearPlane, const float farPlane)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);

			const float f = 1.0f / tanf(fieldOfView / 2.0f);
			result->m_f11 = f / aspectRatio;
			result->m_f22 = f;
			result->m_f33 = (farPlane) / (nearPlane - farPlane);
			result->m_f43 = -1.0f;
			result->m_f34 = (nearPlane * farPlane) / (nearPlane - farPlane);

			return result;
		}
		
		///
		/// @details Creates a left-handed projection matrix for 3D rendering given a few setup parameters.
		///
		/// @param result Where to store the resulting matrix and return.
		/// @param fieldOfView The vertical field of vision for the projection matrix.
		/// @param aspectRatio The aspect ratio of the window/screen or target render.
		/// @param nearPlane The distance to the near plane of the projection matrix.
		/// @param farPlane The distance to the far plane of the projection matrix.
		///
		inline Matrix4* MatrixCreatePerspectiveLH(Matrix4* result, const float fieldOfView, const float aspectRatio, 
			const float nearPlane, const float farPlane)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);

			const float f = 1.0f / tanf(fieldOfView / 2.0f);
			result->m_f11 = f / aspectRatio;
			result->m_f22 = f;
			result->m_f33 = (farPlane) / (farPlane - nearPlane);
			result->m_f43 = 1.0f;
			result->m_f34 = -(nearPlane * farPlane) / (farPlane - nearPlane);

			return result;
		}

		///
		/// @details Creates a right-handed orthographic matrix for 2D rendering given a few setup parameters.  The
		///   Matrix will look something like the following once returned:
		///
		///   2 / width,          0,                0,             0,
		///       0,          2 / height,           0,             0,
		///       0,              0,          1 / (near-far),      0,
		///       0,              0,          near/(near-far)      1
		///
		/// @param result Where to store the resulting matrix and return.
		/// @param width The width of the screen/window/surface available for the 2D rendering.
		/// @param height The height of the screen/window/surface available for the 2D rendering.
		/// @param nearPlane The distance to the near plane of the orthographic matrix. (closest Z value).
		/// @param farPlane The distance to the far plane of the orthographic matrix. (farthest Z value).
		///
		inline Matrix4* MatrixCreateOrthoRH(Matrix4* result, const float width, const float height, 
			const float nearPlane, const float farPlane)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);

			result->m_f11 = 2.0f / width;
			result->m_f22 = 2.0f / height;
			result->m_f33 = 1.0f / (nearPlane - farPlane);
			result->m_f34 = nearPlane / (nearPlane - farPlane);
			result->m_f44 = 1.0f;

			return result;
		}

		///
		/// @details Creates a right-handed orthographic matrix for 2D rendering given a few setup parameters.  The
		///   Matrix will look something like the following once returned:
		///
		///   2 / width,          0,                0,             0,
		///       0,          2 / height,           0,             0,
		///       0,              0,          1 / (far-near),      0,
		///       0,              0,          near/(near-far)      1
		///
		/// @param result Where to store the resulting matrix and return.
		/// @param width The width of the screen/window/surface available for the 2D rendering.
		/// @param height The height of the screen/window/surface available for the 2D rendering.
		/// @param nearPlane The distance to the near plane of the orthographic matrix. (closest Z value).
		/// @param farPlane The distance to the far plane of the orthographic matrix. (farthest Z value).
		///
		inline Matrix4* MatrixCreateOrthoLH(Matrix4* result, const float width, const float height,
			const float nearPlane, const float farPlane)
		{
			memset(result->mComponents, 0, sizeof(float) << 4);

			//matR->m_f11 = 2.0f / fWidth;
			//matR->m_f22 = 2.0f / fHeight;
			//matR->m_f33 = 1 / (fFar - fNear);
			//matR->m_f34 = fNear / (fNear - fFar);
			//matR->m_f44 = 1;

			result->m_f11 = 2.0f / width;
			result->m_f22 = 2.0f / height;
			result->m_f33 = 1.0f / (farPlane - nearPlane);
			result->m_f34 = nearPlane / (nearPlane - farPlane);
			result->m_f44 = 1.0f;

			return result;
		}

		///
		/// @details Adds two Matrix objects together into matrixResult and then the results matrix is returned.
		///
		/// @param result The matrix to fill out and return with the result of the two matrices added together.
		/// @param leftSide The matrix on the left side of the addition operator.
		/// @param rightSide The matrix on the right side of the addition operator.
		///
		inline Matrix4* MatrixAdd(Matrix4 *result, const Matrix4 *leftSide, const Matrix4 *rightSide)
		{
			float *r = result->mComponents;
			const float *m1 = leftSide->mComponents;
			const float *m2 = rightSide->mComponents;

			r[0] = m1[0] + m2[0];			r[1] = m1[1] + m2[1];			r[2] = m1[2] + m2[2];			r[3] = m1[3] + m2[3];
			r[4] = m1[4] + m2[4];			r[5] = m1[5] + m2[5];			r[6] = m1[6] + m2[6];			r[7] = m1[7] + m2[7];
			r[8] = m1[8] + m2[8];			r[9] = m1[9] + m2[9];			r[10] = m1[10] + m2[10];		r[11] = m1[11] + m2[11];
			r[12] = m1[12] + m2[12];		r[13] = m1[13] + m2[13];		r[14] = m1[14] + m2[14];		r[15] = m1[15] + m2[15];
			return result;
		}

		///
		/// @details Subtracts rightSide matrix from the leftSide matrix and stores the result into result before returning.
		///
		/// @param result The matrix to fill out and return with the result of: leftSide - rightSide.
		/// @param leftSide The matrix on the left side of the subtration operator.
		/// @param rightSide The matrix on the right side of the subtraction operator.
		///
		inline Matrix4* MatrixSubtract(Matrix4* result, const Matrix4* leftSide, const Matrix4* rightSide)
		{
			float *r = result->mComponents;
			const float *m1 = leftSide->mComponents;
			const float *m2 = rightSide->mComponents;

			r[0] = m1[0] - m2[0];			r[1] = m1[1] - m2[1];			r[2] = m1[2] - m2[2];			r[3] = m1[3] - m2[3];
			r[4] = m1[4] - m2[4];			r[5] = m1[5] - m2[5];			r[6] = m1[6] - m2[6];			r[7] = m1[7] - m2[7];
			r[8] = m1[8] - m2[8];			r[9] = m1[9] - m2[9];			r[10] = m1[10] - m2[10];		r[11] = m1[11] - m2[11];
			r[12] = m1[12] - m2[12];		r[13] = m1[13] - m2[13];		r[14] = m1[14] - m2[14];		r[15] = m1[15] - m2[15];
			return result;
		}
		
		///
		/// @details Fills the compoments of result Matrix4 object so it would be the transposed of the input Matrix.
		///
		/// @param result The resulting matrix of the transposed operation on input, cannot be the same as the input matrix.
		/// @param input The matrix to get the transposed result of, cannot be the same as the result matrix.
		///
		inline Matrix4* MatrixTranspose(Matrix4* result, const Matrix4* input)
		{
			tb_error_if(result == input, "tbExternalError: Invalid parameter; expected result to be different than input");
			result->m_f11 = input->m_f11;	result->m_f12 = input->m_f21;	result->m_f13 = input->m_f31;	result->m_f14 = input->m_f41;
			result->m_f21 = input->m_f12;	result->m_f22 = input->m_f22;	result->m_f23 = input->m_f32;	result->m_f24 = input->m_f42;
			result->m_f31 = input->m_f13;	result->m_f32 = input->m_f23;	result->m_f33 = input->m_f33;	result->m_f34 = input->m_f43;
			result->m_f41 = input->m_f14;	result->m_f42 = input->m_f24;	result->m_f43 = input->m_f34;	result->m_f44 = input->m_f44;
			return result;
		}
		
		///
		/// @details Multiplies two matrices together and stores the resulting matrix to be returned.
		///
		/// @param result The result matrix which will be the product of leftSide * rightSide when finished.  This
		///   matrix cannot be the same as leftSide or rightSide or an error condition will be triggered.
		/// @param leftSide The matrix on the left side of the multiplication operation. This cannot be the same as the
		///   result matrix or an error condition will be triggered.
		/// @param rightSide The matrix on the right side of the multiplication operation. This cannot be the same as 
		///   the result matrix or an error condition will be triggered.
		///
		inline Matrix4* MatrixMultiply(Matrix4* result, const Matrix4* leftSide, const Matrix4* rightSide)
		{
			tb_error_if(result == leftSide || result == rightSide, "tbExternalError: Invalid parameter; expected result to be different than leftSide and rightSide");
			result->m_f11 = (leftSide->m_f11 * rightSide->m_f11) + (leftSide->m_f21 * rightSide->m_f12) + (leftSide->m_f31 * rightSide->m_f13) + (leftSide->m_f41 * rightSide->m_f14);
			result->m_f21 = (leftSide->m_f11 * rightSide->m_f21) + (leftSide->m_f21 * rightSide->m_f22) + (leftSide->m_f31 * rightSide->m_f23) + (leftSide->m_f41 * rightSide->m_f24);
			result->m_f31 = (leftSide->m_f11 * rightSide->m_f31) + (leftSide->m_f21 * rightSide->m_f32) + (leftSide->m_f31 * rightSide->m_f33) + (leftSide->m_f41 * rightSide->m_f34);
			result->m_f41 = (leftSide->m_f11 * rightSide->m_f41) + (leftSide->m_f21 * rightSide->m_f42) + (leftSide->m_f31 * rightSide->m_f43) + (leftSide->m_f41 * rightSide->m_f44);
			result->m_f12 = (leftSide->m_f12 * rightSide->m_f11) + (leftSide->m_f22 * rightSide->m_f12) + (leftSide->m_f32 * rightSide->m_f13) + (leftSide->m_f42 * rightSide->m_f14);
			result->m_f22 = (leftSide->m_f12 * rightSide->m_f21) + (leftSide->m_f22 * rightSide->m_f22) + (leftSide->m_f32 * rightSide->m_f23) + (leftSide->m_f42 * rightSide->m_f24);
			result->m_f32 = (leftSide->m_f12 * rightSide->m_f31) + (leftSide->m_f22 * rightSide->m_f32) + (leftSide->m_f32 * rightSide->m_f33) + (leftSide->m_f42 * rightSide->m_f34);
			result->m_f42 = (leftSide->m_f12 * rightSide->m_f41) + (leftSide->m_f22 * rightSide->m_f42) + (leftSide->m_f32 * rightSide->m_f43) + (leftSide->m_f42 * rightSide->m_f44);
			result->m_f13 = (leftSide->m_f13 * rightSide->m_f11) + (leftSide->m_f23 * rightSide->m_f12) + (leftSide->m_f33 * rightSide->m_f13) + (leftSide->m_f43 * rightSide->m_f14);
			result->m_f23 = (leftSide->m_f13 * rightSide->m_f21) + (leftSide->m_f23 * rightSide->m_f22) + (leftSide->m_f33 * rightSide->m_f23) + (leftSide->m_f43 * rightSide->m_f24);
			result->m_f33 = (leftSide->m_f13 * rightSide->m_f31) + (leftSide->m_f23 * rightSide->m_f32) + (leftSide->m_f33 * rightSide->m_f33) + (leftSide->m_f43 * rightSide->m_f34);
			result->m_f43 = (leftSide->m_f13 * rightSide->m_f41) + (leftSide->m_f23 * rightSide->m_f42) + (leftSide->m_f33 * rightSide->m_f43) + (leftSide->m_f43 * rightSide->m_f44);
			result->m_f14 = (leftSide->m_f14 * rightSide->m_f11) + (leftSide->m_f24 * rightSide->m_f12) + (leftSide->m_f34 * rightSide->m_f13) + (leftSide->m_f44 * rightSide->m_f14);
			result->m_f24 = (leftSide->m_f14 * rightSide->m_f21) + (leftSide->m_f24 * rightSide->m_f22) + (leftSide->m_f34 * rightSide->m_f23) + (leftSide->m_f44 * rightSide->m_f24);
			result->m_f34 = (leftSide->m_f14 * rightSide->m_f31) + (leftSide->m_f24 * rightSide->m_f32) + (leftSide->m_f34 * rightSide->m_f33) + (leftSide->m_f44 * rightSide->m_f34);
			result->m_f44 = (leftSide->m_f14 * rightSide->m_f41) + (leftSide->m_f24 * rightSide->m_f42) + (leftSide->m_f34 * rightSide->m_f43) + (leftSide->m_f44 * rightSide->m_f44);
			return result;
		}

		///
		/// @details Multiplies a vector and matrix together to transform the inputVector into the space described by
		///   the inputMatrix storing and returning the result.
		///
		/// @param result The vector to hold the result of the transformed inputVector, must be a different vector
		///   object or an error condition will be triggered.
		/// @param inputVector The vector to transform by the input matrix, must not be the same as result.
		/// @param inputMatrix The transform matrix to multiply inputVector by.
		///
		inline Vector3* Vector3MatrixMultiply(Vector3* result, const Vector3* inputVector, const Matrix3* inputMatrix)
		{
			tb_error_if(result == inputVector, "tbExternalError: Invalid parameter; expected result to be different than inputVector.");
			result->x = (inputVector->x * inputMatrix->m_f11) + (inputVector->y * inputMatrix->m_f12) + (inputVector->z * inputMatrix->m_f13);
			result->y = (inputVector->x * inputMatrix->m_f21) + (inputVector->y * inputMatrix->m_f22) + (inputVector->z * inputMatrix->m_f23);
			result->z = (inputVector->x * inputMatrix->m_f31) + (inputVector->y * inputMatrix->m_f32) + (inputVector->z * inputMatrix->m_f33);
			return result;
		}

		///
		/// @details Multiplies a vector and matrix together to transform the inputVector into the space described by
		///   the inputMatrix storing and returning the result.
		///
		/// @param result The vector to hold the result of the transformed inputVector, must be a different vector
		///   object or an error condition will be triggered.
		/// @param inputVector The vector to transform by the input matrix, must not be the same as result.
		/// @param inputMatrix The transform matrix to multiply inputVector by.
		///
		inline Vector4* Vector4MatrixMultiply(Vector4* result, const Vector4* inputVector, const Matrix4* inputMatrix)
		{
			tb_error_if(result == inputVector , "tbExternalError: Invalid parameter; expected result to be different than inputVector.");
			result->x = (inputVector->x * inputMatrix->m_f11) + (inputVector->y * inputMatrix->m_f12) + (inputVector->z * inputMatrix->m_f13) + (inputVector->w * inputMatrix->m_f14);
			result->y = (inputVector->x * inputMatrix->m_f21) + (inputVector->y * inputMatrix->m_f22) + (inputVector->z * inputMatrix->m_f23) + (inputVector->w * inputMatrix->m_f24);
			result->z = (inputVector->x * inputMatrix->m_f31) + (inputVector->y * inputMatrix->m_f32) + (inputVector->z * inputMatrix->m_f33) + (inputVector->w * inputMatrix->m_f34);
			result->w = (inputVector->x * inputMatrix->m_f41) + (inputVector->y * inputMatrix->m_f42) + (inputVector->z * inputMatrix->m_f43) + (inputVector->w * inputMatrix->m_f44);
			return result;
		}

		/*~**********************************************************************************************************************
		*	~Function:			"Vector3TransformCoord"																			*
		*	~Modified By:		Tim Beaudet																						*
		*	~Last Updated:		11/8/2009																						*
		*																														*
		*	~Purpose:			Multiplies a vector and a matrix and returns the results in vR.  This multiplies the 3component	*
		*						vector v1 by the matrix.  v1 is taken as a coordinate, so its W component is set to 1.			*
		*																														*
		*	~In:				vR		The results of the multiplication of the vector and the matrix.							*
		*						mat		The matrix to multiply.																	*
		*						v1		The vector to multiply, can NOT be vR.													*
		*																														*
		*	~Out:				The resulting transformed coordinate.  Notice that W component is set to 1.						*
		**********************************************************************************************************************~*/

		///
		/// @details Multiplies a vector and matrix together to transform the inputVector into the space described by
		///   the inputMatrix storing and returning the result.  This behaves identical to Vector4MatrixMultiply except
		///   a Vector3 does not have a w component and saves upon that final round of multiplications.
		///
		/// @param result The vector to hold the result of the transformed inputVector, must be a different vector
		///   object or an error condition will be triggered.
		/// @param inputVector The vector to transform by the input matrix, must not be the same as result.
		/// @param inputMatrix The transform matrix to multiply inputVector by.
		///
		inline Vector3* Vector3TransformCoord(Vector3* result, const Vector3* inputVector, const Matrix4* inputMatrix)
		{
			tb_error_if(result == inputVector, "tbExternalError: Invalid parameter; expected result to be different than inputVector");
			result->x = (inputVector->x * inputMatrix->m_f11) + (inputVector->y * inputMatrix->m_f12) + (inputVector->z * inputMatrix->m_f13) + inputMatrix->m_f14;
			result->y = (inputVector->x * inputMatrix->m_f21) + (inputVector->y * inputMatrix->m_f22) + (inputVector->z * inputMatrix->m_f23) + inputMatrix->m_f24;
			result->z = (inputVector->x * inputMatrix->m_f31) + (inputVector->y * inputMatrix->m_f32) + (inputVector->z * inputMatrix->m_f33) + inputMatrix->m_f34;
			return result;
		}

		/*~**********************************************************************************************************************
		*	~Function:			"Vector3TransformNormal"																		*
		*	~Modified By:		Tim Beaudet																						*
		*	~Last Updated:		11/8/2009																						*
		*																														*
		*	~Purpose:			Multiplies a vector and a matrix and returns the results in vR.  This multiplies the 3component	*
		*						vector v1 by the matrix.  v1 is taken as a coordinate, so its W component is set to 0.			*
		*																														*
		*	~In:				vR		The results of the multiplication of the vector and the matrix.							*
		*						mat		The matrix to multiply.																	*
		*						v1		The vector to multiply, can NOT be vR.													*
		*																														*
		*	~Out:				The resulting transformed normal or directional vector.  Notice that W component is set to 0.	*
		**********************************************************************************************************************~*/

		///
		/// @details Multiplies a vector and matrix together to transform the inputVector into the space described by
		///   the inputMatrix storing and returning the result.  This behaves differently than Vector3TransformCoord
		///   because it does not consider the translation of the matrix, only the rotation.
		///
		/// @param result The vector to hold the result of the transformed inputVector, must be a different vector
		///   object or an error condition will be triggered.
		/// @param inputVector The vector to transform by the input matrix, must not be the same as result.
		/// @param inputMatrix The transform matrix to multiply inputVector by.
		///
		inline Vector3* Vector3TransformNormal(Vector3* result, const Vector3* inputVector, const Matrix4* inputMatrix)
		{
			tb_error_if(result == inputVector, "tbExternalError: Invalid parameter; expected result to be different than inputVector");
			result->x = (inputVector->x * inputMatrix->m_f11) + (inputVector->y * inputMatrix->m_f12) + (inputVector->z * inputMatrix->m_f13);
			result->y = (inputVector->x * inputMatrix->m_f21) + (inputVector->y * inputMatrix->m_f22) + (inputVector->z * inputMatrix->m_f23);
			result->z = (inputVector->x * inputMatrix->m_f31) + (inputVector->y * inputMatrix->m_f32) + (inputVector->z * inputMatrix->m_f33);
			return result;
		}

		///
		/// @details Computes the determinant of the 3x3 matrix components and returns the result.  Essentially a supporting
		///   function for MatrixDeterminant and ComputeInverse below for the 4x4 matrix.
		///
		inline float Matrix3x3Determinant(const float f11, const float f12, const float f13, const float f21, const float f22, const float f23, const float f31, const float f32, const float f33)
		{
			return f11 * (f22 * f33 - f32 * f23) - f12 * (f21 * f33 - f31 * f23) + f13 * (f21 * f32 - f31 * f22);
		}

		///
		/// @details Computes the determinant of the 4x4 matrix by breaking it down into 3x3 determinants and returns 
		///   the resulting scalar determinant.
		///
		inline float MatrixDeterminant(const Matrix4* input)
		{
			return  input->m_f11 * Matrix3x3Determinant(input->m_f22, input->m_f32, input->m_f42, input->m_f23, input->m_f33, input->m_f43, input->m_f24, input->m_f34, input->m_f44) -
					input->m_f21 * Matrix3x3Determinant(input->m_f12, input->m_f32, input->m_f42, input->m_f13, input->m_f33, input->m_f43, input->m_f14, input->m_f34, input->m_f44) +
					input->m_f31 * Matrix3x3Determinant(input->m_f12, input->m_f22, input->m_f42, input->m_f13, input->m_f23, input->m_f43, input->m_f14, input->m_f24, input->m_f44) -
					input->m_f41 * Matrix3x3Determinant(input->m_f12, input->m_f22, input->m_f32, input->m_f13, input->m_f23, input->m_f33, input->m_f14, input->m_f24, input->m_f34);
		}
		
		///
		/// @details Computes the inverse matrix that when multiplied against the initial input would result in an 
		///   identity matrix.
		///
		/// @param result The matrix to store and return the computed inverse matrix of input.  This cannot be the same
		///   matrix as the input matrix or an error condition will be triggered.
		/// @param input The matrix to compute the inverse of, cannot be the same as the result matrix or an error
		///   condition will be triggered.
		///
		/// @note this is a fairly heavy lifter so use lightly.
		///
		inline Matrix4* MatrixComputeInverse(Matrix4* result, const Matrix4* input)
		{
			tb_error_if(result == input, "tbExternalError: Invalid parameter, expected result to be different than input.");
			const float determinant = MatrixDeterminant(input);
			if (true == IsZero(determinant))
			{
				*result = *input;
				return result;
			}

			const float inverseDeterminant = 1.0f / determinant;

			result->m_f11 = (Matrix3x3Determinant (input->m_f22, input->m_f32, input->m_f42, input->m_f23, input->m_f33, input->m_f43, input->m_f24, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f21 = (-Matrix3x3Determinant(input->m_f21, input->m_f31, input->m_f41, input->m_f23, input->m_f33, input->m_f43, input->m_f24, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f31 = (Matrix3x3Determinant (input->m_f21, input->m_f31, input->m_f41, input->m_f22, input->m_f32, input->m_f42, input->m_f24, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f41 = (-Matrix3x3Determinant(input->m_f21, input->m_f31, input->m_f41, input->m_f22, input->m_f32, input->m_f42, input->m_f23, input->m_f33, input->m_f43)) * inverseDeterminant;
			result->m_f12 = (-Matrix3x3Determinant(input->m_f12, input->m_f32, input->m_f42, input->m_f13, input->m_f33, input->m_f43, input->m_f14, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f22 = (Matrix3x3Determinant (input->m_f11, input->m_f31, input->m_f41, input->m_f13, input->m_f33, input->m_f43, input->m_f14, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f32 = (-Matrix3x3Determinant(input->m_f11, input->m_f31, input->m_f41, input->m_f12, input->m_f32, input->m_f42, input->m_f14, input->m_f34, input->m_f44)) * inverseDeterminant;
			result->m_f42 = (Matrix3x3Determinant (input->m_f11, input->m_f31, input->m_f41, input->m_f12, input->m_f32, input->m_f42, input->m_f13, input->m_f33, input->m_f43)) * inverseDeterminant;
			result->m_f13 = (Matrix3x3Determinant (input->m_f12, input->m_f22, input->m_f42, input->m_f13, input->m_f23, input->m_f43, input->m_f14, input->m_f24, input->m_f44)) * inverseDeterminant;
			result->m_f23 = (-Matrix3x3Determinant(input->m_f11, input->m_f21, input->m_f41, input->m_f13, input->m_f23, input->m_f43, input->m_f14, input->m_f24, input->m_f44)) * inverseDeterminant;
			result->m_f33 = (Matrix3x3Determinant (input->m_f11, input->m_f21, input->m_f41, input->m_f12, input->m_f22, input->m_f42, input->m_f14, input->m_f24, input->m_f44)) * inverseDeterminant;
			result->m_f43 = (-Matrix3x3Determinant(input->m_f11, input->m_f21, input->m_f41, input->m_f12, input->m_f22, input->m_f42, input->m_f13, input->m_f23, input->m_f43)) * inverseDeterminant;
			result->m_f14 = (-Matrix3x3Determinant(input->m_f12, input->m_f22, input->m_f32, input->m_f13, input->m_f23, input->m_f33, input->m_f14, input->m_f24, input->m_f34)) * inverseDeterminant;
			result->m_f24 = (Matrix3x3Determinant (input->m_f11, input->m_f21, input->m_f31, input->m_f13, input->m_f23, input->m_f33, input->m_f14, input->m_f24, input->m_f34)) * inverseDeterminant;
			result->m_f34 = (-Matrix3x3Determinant(input->m_f11, input->m_f21, input->m_f31, input->m_f12, input->m_f22, input->m_f32, input->m_f14, input->m_f24, input->m_f34)) * inverseDeterminant;
			result->m_f44 = (Matrix3x3Determinant (input->m_f11, input->m_f21, input->m_f31, input->m_f12, input->m_f22, input->m_f32, input->m_f13, input->m_f24, input->m_f33)) * inverseDeterminant;

			return result;
		}

		namespace Unstable
		{
			inline Matrix4 Translate(const Matrix4& originalMatrix, const Vector3& translation)
			{
				Matrix4 outputMatrix(kSkipInitialization);
				Matrix4 translationMatrix(kSkipInitialization);

				MatrixCreateTranslation(&translationMatrix, &translation);
				const Matrix4 translationTransposed(translationMatrix.GetTransposed());
				MatrixMultiply(&outputMatrix, &originalMatrix, &translationTransposed);

				return outputMatrix;
			}

			inline Matrix4 Translate(const Matrix4& originalMatrix, float x, float y, float z)
			{
				return Translate(originalMatrix, tbMath::Vector3(x, y, z));
			}

			///
			/// @note rotates clockwise when looking down the axis provided.
			///
			inline Matrix4 Rotate(const Matrix4& originalMatrix, const Vector3& aroundAxis, const float rotationInDegrees)
			{
				Matrix4 outputMatrix(kSkipInitialization);
				Matrix4 rotationMatrix(kSkipInitialization);

				//Negating here for positive rotationInDegrees to spin clockwise around the axis.
				MatrixCreateRotationA(&rotationMatrix, &aroundAxis, rotationInDegrees);
				const Matrix4 rotationTransposed(rotationMatrix.GetTransposed());
				MatrixMultiply(&outputMatrix, &originalMatrix, &rotationTransposed);

				return outputMatrix;
			}

			inline Matrix4 Rotate(const size_t& axisIndex, const Matrix4& originalMatrix, const float rotationInDegrees)
			{
				return Rotate(originalMatrix, *originalMatrix.GetBasis(axisIndex), rotationInDegrees);
			}

			inline Matrix4 RotateWorldSpace(const size_t& axisIndex, const Matrix4& originalMatrix, const float rotationInDegrees)
			{
				Matrix4 outputMatrix(kSkipInitialization);
				Matrix4 rotationMatrix(kSkipInitialization);

				//Negating here for positive rotationInDegrees to spin clockwise around the axis.
				MatrixCreateRotationA(&rotationMatrix, Matrix4::kIdentity.GetBasis(axisIndex), rotationInDegrees);
				const Matrix4 rotationTransposed(rotationMatrix.GetTransposed());
				MatrixMultiply(&outputMatrix, &originalMatrix, &rotationTransposed);

				return outputMatrix;
			}

			inline Matrix4 Scale(const Matrix4& originalMatrix, const Vector3& scale)
			{
				Matrix4 outputMatrix(kSkipInitialization);
				Matrix4 scalingMatrix(kSkipInitialization);

				MatrixCreateScale(&scalingMatrix, &scale);
				MatrixMultiply(&outputMatrix, &originalMatrix, &scalingMatrix);

				return outputMatrix;
			}

			inline Matrix4 Scale(const Matrix4& originalMatrix, float x, float y, float z)
			{
				return Scale(originalMatrix, Vector3(x, y, z));
			}

		}; /* namespace Unstable */

	}; /* namespace Math */
}; /* namespace TurtleBrains */

namespace tbMath = TurtleBrains::Math;

#endif /* _TurtleBrains_Matrix_h_ */
