///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tbu_bounding_volume.h"
#include "../../core/tb_error.h"

//--------------------------------------------------------------------------------------------------------------------//

tbMath::Unstable::BoundingVolume::BoundingVolume(const float radius) :
	mWidth(radius),
	mHeight(-1.0f)
{
	tb_error_if(radius <= 0.0f, "tbExternalError: Expected parameter radius(%.3f) to be greater than 0.0", radius);
}

//--------------------------------------------------------------------------------------------------------------------//

tbMath::Unstable::BoundingVolume::BoundingVolume(const float width, const float height) :
	mWidth(width),
	mHeight(height)
{
	tb_error_if(width <= 0.0f, "tbExternalError: Expected parameter width(%.3f) to be greater than 0.0", width);
	tb_error_if(height <= 0.0f, "tbExternalError: Expected parameter height(%.3f) to be greater than 0.0", height);
}

//--------------------------------------------------------------------------------------------------------------------//

tbMath::Unstable::BoundingVolume::~BoundingVolume(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbMath::Unstable::BoundingVolume::IsCircle(void) const
{
	return (mHeight <= 0.0f) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbMath::Unstable::BoundingVolume::IsBox(void) const
{
	return (mHeight > 0.0f) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//


bool tbMath::Unstable::BoundingVolume::CheckCollisionWith(const tbMath::Vector2& objectPosition, const tbMath::Vector2& testPoint) const
{
	if (true == IsCircle())
	{
		tbMath::Vector2 toTestPoint = (objectPosition - testPoint);
		if (toTestPoint.MagnitudeSquared() < (mWidth * mWidth))
		{ //Point in Circle
			return true;
		}
	}
	else //if (true == IsBox())
	{
		const float halfWidth = mWidth * 0.5f;
		const float halfHeight = mHeight * 0.5f;
		if (testPoint.x > objectPosition.x - halfWidth && testPoint.x < objectPosition.x + halfWidth &&
				testPoint.y > objectPosition.y - halfHeight && testPoint.y < objectPosition.y + halfHeight)
		{	//Point in AABB
			return true;
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

//
// Circle to Circle Collision
//
bool CheckCollisionOf(const tbMath::Vector2& center, const float radius, const tbMath::Vector2& otherCenter, const float otherRadius)
{
	tbMath::Vector2 toOtherCenter(otherCenter - center);
//	if (toOtherCenter.MagnitudeSquared() < (radius * radius) + (otherRadius * otherRadius))
	if (toOtherCenter.MagnitudeSquared() < (radius + otherRadius) * (radius + otherRadius))
	{
		return true;
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

//
// AABB to AABB Collision
//
bool CheckCollisionOf(const tbMath::Vector2& center, const float halfWidth, const float halfHeight,
	const tbMath::Vector2& otherCenter, const float otherHalfWidth, const float otherHalfHeight)
{
	if (otherCenter.x - otherHalfWidth > center.x + halfWidth) { return false; }
	if (otherCenter.x + otherHalfWidth < center.x - halfWidth) { return false; }
	if (otherCenter.y - otherHalfHeight > center.y + halfHeight) { return false; }
	if (otherCenter.y + otherHalfHeight < center.y - halfHeight) { return false; }
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

//
// AABB to Circle Collision
//
bool CheckCollisionOf(const tbMath::Vector2& center, const float halfWidth, const float halfHeight,
											const tbMath::Vector2& otherCenter, const float otherRadius)
{	//http://stackoverflow.com/questions/401847/circle-rectangle-collision-detection-intersection
	float circleDistanceX = fabs(otherCenter.x - center.x);
	float circleDistanceY = fabs(otherCenter.y - center.y);

	if (circleDistanceX > (halfWidth + otherRadius)) { return false; }
	if (circleDistanceY > (halfHeight + otherRadius)) { return false; }

	if (circleDistanceX <= halfWidth) { return true; }
	if (circleDistanceY <= halfHeight) { return true; }

	float cornerDistanceSquared(((circleDistanceX - halfWidth) *  (circleDistanceX - halfWidth)) +
		((circleDistanceY - halfHeight) * (circleDistanceY - halfHeight)));

	if (cornerDistanceSquared <= (otherRadius * otherRadius)) { return true; }

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbMath::Unstable::BoundingVolume::CheckCollisionWith(const tbMath::Vector2& objectPosition, const BoundingVolume& testVolume,
	const tbMath::Vector2& testVolumePosition) const
{
	if (true == IsCircle())
	{
		if (true == testVolume.IsCircle())
		{
			return CheckCollisionOf(objectPosition, mWidth, testVolumePosition, testVolume.mWidth);
		}
		else //if (true == testVolume.IsBox())
		{
			return CheckCollisionOf(testVolumePosition, testVolume.mWidth * 0.5f, testVolume.mHeight * 0.5f, objectPosition, mWidth);
		}
	}
	else //if (true == IsBox())
	{
		if (true == testVolume.IsCircle())
		{
			return CheckCollisionOf(objectPosition, mWidth * 0.5f, mHeight * 0.5f, testVolumePosition, testVolume.mWidth);
		}
		else //if (true == testVolume.IsBox())
		{
			return CheckCollisionOf(objectPosition, mWidth * 0.5f, mHeight * 0.5f, testVolumePosition, testVolume.mWidth * 0.5f, testVolume.mHeight * 0.5f);
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

bool tbMath::Unstable::BoundingVolume::ResolveCollisionWithStatic(tbMath::Vector2& objectPosition, const BoundingVolume& staticVolume, const tbMath::Vector2& staticVolumePosition) const
{
	if (false == IsBox() || false == staticVolume.IsBox())
	{
		return false;
	}

	tbMath::Vector2 thisMinimum(objectPosition.x - (mWidth / 2.0f), objectPosition.y - (mHeight / 2.0f));
	tbMath::Vector2 thisMaximum(objectPosition.x + (mWidth / 2.0f), objectPosition.y + (mHeight / 2.0f));
	tbMath::Vector2 staticMinimum(staticVolumePosition.x - (staticVolume.mWidth / 2.0f), staticVolumePosition.y - (staticVolume.mHeight / 2.0f));
	tbMath::Vector2 staticMaximum(staticVolumePosition.x + (staticVolume.mWidth / 2.0f), staticVolumePosition.y + (staticVolume.mHeight / 2.0f));

	tbMath::Vector2 minimumTranslation(tbMath::Vector2::kZero);

	const float left(staticMinimum.x - thisMaximum.x);
	const float right(staticMaximum.x - thisMinimum.x);
	const float top(staticMinimum.y - thisMaximum.y);
	const float bottom(staticMaximum.y - thisMinimum.y);

	tb_error_if(left > 0.0f || right < 0.0f || top > 0.0f || bottom < 0.0f, "tbExternalError: The bounding volumes do not intersect.");

	minimumTranslation.x = (fabs(left) < right) ? left : right;
	minimumTranslation.y = (fabs(top) < bottom) ? top : bottom;

	if (fabs(minimumTranslation.x) < fabs(minimumTranslation.y))
	{
		minimumTranslation.y = 0.0f;
	}
	else
	{
		minimumTranslation.x = 0.0f;
	}

	if (fabs(minimumTranslation.x) < tbMath::kTolerance || fabs(minimumTranslation.y) < tbMath::kTolerance)
	{
			printf("mtd: %.2f, %.2f\n", minimumTranslation.x, minimumTranslation.y);
		objectPosition += minimumTranslation;
	}

	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

