///
/// @file
/// @details A way to store and set results, like current/best score for the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "game_results.h"

namespace GameResults
{

	template <typename Type> class ValueTracker
	{
	public:
		ValueTracker(const Type& initialValue) :
			mInitialValue(initialValue),
			mCurrentValue(initialValue),
			mMinimumValue(initialValue),
			mMaximumValue(initialValue)
		{
		}

		void SetCurrentValue(const Type& newValue)
		{
			mCurrentValue = newValue;
			if (newValue < GetMinimumValue()) { mMinimumValue = newValue; }
			if (GetMaximumValue() < newValue) { mMaximumValue = newValue; }
		}

		void IncrementValue(const Type& byAmount)
		{
			SetCurrentValue(mCurrentValue + byAmount);
		}

		void ResetTracker(bool includingTrackedRange)
		{
			mCurrentValue = mInitialValue;
			if (true == includingTrackedRange)
			{
				mMinimumValue = mInitialValue;
				mMaximumValue = mInitialValue;
			}
		}

		const Type& GetCurrentValue(void) const { return mCurrentValue; }
		const Type& GetMinimumValue(void) const { return mMinimumValue; }
		const Type& GetMaximumValue(void) const { return mMaximumValue; }

	private:
		const Type mInitialValue;
		Type mCurrentValue;
		Type mMinimumValue;
		Type mMaximumValue;
	};

	ValueTracker<uint32_t> theScore(0);
	ValueTracker<uint32_t> theHoursSpentInRoom(0);
	ValueTracker<uint32_t> theMonsterKillCount(0);
	bool theWinFlag(false);

}; /* namespace GameResults */

//--------------------------------------------------------------------------------------------------------------------//

void GameResults::ResetGameResults(bool includingBest)
{
	theScore.ResetTracker(includingBest);
	theHoursSpentInRoom.ResetTracker(includingBest);
	theMonsterKillCount.ResetTracker(includingBest);
	theWinFlag = false;
}

//--------------------------------------------------------------------------------------------------------------------//

uint32_t GameResults::GetCurrentScore(void)
{
	return theScore.GetCurrentValue();
}

//--------------------------------------------------------------------------------------------------------------------//

uint32_t GameResults::GetBestScore(void)
{
	return theScore.GetMaximumValue();
}

//--------------------------------------------------------------------------------------------------------------------//

void GameResults::IncrementCurrentScore(const uint32_t& increaseAmount)
{
	theScore.IncrementValue(increaseAmount);
}

//--------------------------------------------------------------------------------------------------------------------//

void GameResults::IncrementHoursSpentInRoom(void) { theHoursSpentInRoom.IncrementValue(1); }
uint32_t GameResults::GetHoursSpentInRoom(void) { return theHoursSpentInRoom.GetCurrentValue(); }
uint32_t GameResults::GetMostHoursSpentInRoom(void) { return theHoursSpentInRoom.GetMaximumValue(); }

//--------------------------------------------------------------------------------------------------------------------//

void GameResults::IncrementMonsterKillCount(void) { theMonsterKillCount.IncrementValue(1); }
uint32_t GameResults::GetMonsterKillCount(void) { return theMonsterKillCount.GetCurrentValue(); }
uint32_t GameResults::GetBestMonsterKillCount(void) { return theMonsterKillCount.GetMaximumValue(); }

//--------------------------------------------------------------------------------------------------------------------//

void GameResults::SetWinFlag(bool didWin) { theWinFlag = didWin; }
bool GameResults::GetWinFlag(void) { return theWinFlag; }

//--------------------------------------------------------------------------------------------------------------------//
