///
/// @file
/// @details An entity within the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "haunted_room_entity.h"
#include "game_results.h"

const tbGame::GameTimer kMaximumDealDamageTime(2500);
const tbGame::GameTimer kMaximumKillPlayerTime(4000);

//--------------------------------------------------------------------------------------------------------------------//

HauntedRoomEntity::HauntedRoomEntity(void) :
	tbGame::Entity("HauntedRoomEntity"),
	mCombatTargets(),
	mNumberMonstersForNextCombat(0),
	mCanKillPlayer(false),
	mCanTargetMonster(false),
	mIsMonsterTargetted(false),
	mTargettedMonsterType(kMonsterInvalid),
	mTargettedMonster(nullptr),
	mPreppedSpellSchool(SpellSchool::kInvalid),
	mCastedSpellSchool(SpellSchool::kInvalid),
	mDealDamageTimer(tbGame::GameTimer::Zero()),
	mKillPlayerTimer(tbGame::GameTimer::Zero())

{
	AddGraphic(new tbGraphics::Sprite("data/haunted_room_backdrop.png"));
}

//--------------------------------------------------------------------------------------------------------------------//

HauntedRoomEntity::~HauntedRoomEntity(void)
{
	ResetToEmptyRoom(); //Happens to remove dynamic memory.
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::ResetToEmptyRoom(void)
{
	for (auto itr = mCombatTargets.begin(); itr != mCombatTargets.end(); /* in loop */)
	{
		MonsterEntity& monster(*(*itr));
		monster.GetEntityManager()->RemoveEntity(&monster);
		itr = mCombatTargets.erase(itr);
	}

	mNumberMonstersForNextCombat = 0;
	mCanKillPlayer = true;
	mCanTargetMonster = false;
	ClearTargettedMonster();

	mPreppedSpellSchool = SpellSchool::kInvalid;
	mCastedSpellSchool = SpellSchool::kInvalid;

	mDealDamageTimer = tbGame::GameTimer::Zero();
	mKillPlayerTimer = tbGame::GameTimer::Zero();
}

//--------------------------------------------------------------------------------------------------------------------//

bool HauntedRoomEntity::IsInCombat(void) const
{
	return !mCombatTargets.empty();
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::EnterCombat(const int hoursSpentInRoom)
{
	tb_unused(hoursSpentInRoom);

	//Generate the monsters/targets in the room for this hour.
	const size_t kNumberOfRolls(NumberMonstersInRoomForNextCombat());
	for (size_t diceIndex(0); diceIndex < kNumberOfRolls; ++diceIndex)
	{
		const MonsterType& monsterType(static_cast<MonsterType>(tbMath::RandomInt() % (kMaximumMonsterTypes - 1)));
		MonsterEntity* combatTarget = new MonsterEntity(*this, monsterType);
		//MonsterEntity* combatTarget = new MonsterEntity(*this, MonsterType::kMonsterSkeleton); //For maximized testing
		if (diceIndex >= 16)
		{
			combatTarget->SetPosition(148.0f + (130.0f * (diceIndex - 16)), 496.0f);
		}
		else if (diceIndex >= 8)
		{
			combatTarget->SetPosition(148.0f + (130.0f * (diceIndex - 8)), 696.0f);
		}
		else
		{
			combatTarget->SetPosition(148.0f + (130.0f * diceIndex), 596.0f);
		}

		mCombatTargets.push_back(combatTarget);
		GetEntityManager()->AddEntity(combatTarget);
	}

	SetCanTargetMonster(false);
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::SetCanTargetMonster(const bool canTargetMonster)
{
	mCanTargetMonster = canTargetMonster;
}

//--------------------------------------------------------------------------------------------------------------------//

bool HauntedRoomEntity::IsMonsterTargetted(void) const
{
	return mIsMonsterTargetted;
}

//--------------------------------------------------------------------------------------------------------------------//

bool HauntedRoomEntity::IsMonsterTargetted(const MonsterEntity& monster, const SpellSchool& bySpell) const
{
	if (false == IsMonsterTargetted())
	{
		return false;
	}

	tb_error_if(kMonsterInvalid == mTargettedMonsterType, "Unexpected...");

	if (mTargettedMonster != &monster)
	{
		const size_t bySpellSchoolIndex(SpellSchoolToIndex(bySpell));
		const size_t targettedMonsterTypeIndex(mTargettedMonster->GetMonsterType());
		const size_t monsterTypeIndex(monster.GetMonsterType());

		if ((bySpellSchoolIndex == monsterTypeIndex && bySpellSchoolIndex == targettedMonsterTypeIndex) ||
			(SpellSchool::kMind == bySpell && monster.GetMonsterType() == mTargettedMonster->GetMonsterType()))
		{
			return true;
		}

		return false;
	}

	return true;	
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::SetTargetToMonster(MonsterEntity& targettedMonster)
{
	tb_error_if(false == mCanTargetMonster, "Expected to target monsters only when possible.");

	mIsMonsterTargetted = true;
	mTargettedMonster = &targettedMonster;
	mTargettedMonsterType = targettedMonster.GetMonsterType();
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::ClearTargettedMonster(void)
{
	mIsMonsterTargetted = false;
	mTargettedMonster = nullptr;
	mTargettedMonsterType = kMonsterInvalid;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::ShowAllAvailableTargets(void)
{
	for (MonsterEntity* monster : mCombatTargets)
	{
		if (IsMonsterTargetted(*monster, mPreppedSpellSchool))
		{
			monster->ShowAsSelected();
		}
		else
		{
			monster->ShowAsAvailable();
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::ShowNoAvailableTargets(void)
{
	for (MonsterEntity* monster : mCombatTargets)
	{
		monster->ShowAsUnavailable();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::SetPreppedSpellSchool(const SpellSchool& spellSchool)
{
	mPreppedSpellSchool = spellSchool;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::SetCastedSpellSchool(const SpellSchool& spellSchool)
{
	mCastedSpellSchool = spellSchool;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::SpellCastedOnTargets(void)
{
	mDealDamageTimer = kMaximumDealDamageTime;
}

//--------------------------------------------------------------------------------------------------------------------//

bool HauntedRoomEntity::IsDamageBeingDealt(void) const
{
	return mDealDamageTimer.IsZero() ? false : true;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::MonstersAttackStudent(void)
{
	if (true == mCanKillPlayer && true == mKillPlayerTimer.IsZero())
	{
		mCanKillPlayer = false;
		mKillPlayerTimer = kMaximumKillPlayerTime;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

bool HauntedRoomEntity::IsKillingStudent(void) const
{
	return mKillPlayerTimer.IsZero() ? false : true;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::RecalculateNextCombatMonsterCount(void)
{
	const int hoursSpent(GameResults::GetHoursSpentInRoom() + 1);
	mNumberMonstersForNextCombat = hoursSpent + tbMath::RandomInt(0, hoursSpent);
//	mNumberMonstersForNextCombat = hoursSpent + hoursSpent; //For maximized testing
}

//--------------------------------------------------------------------------------------------------------------------//

int HauntedRoomEntity::NumberMonstersInRoomForNextCombat(void) const
{
	return mNumberMonstersForNextCombat;
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnAdded(void)
{
	tbGame::Entity::OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnRemoved(void)
{
	tbGame::Entity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnSimulate(void)
{
	tbGame::Entity::OnSimulate();

	if (false == mDealDamageTimer.IsZero())
	{
		if (false == mDealDamageTimer.DecrementStep())
		{
			//TODO: Effects...
			for (auto itr = mCombatTargets.begin(); itr != mCombatTargets.end(); /* in loop */)
			{
				MonsterEntity& monster(*(*itr));
				if (IsMonsterTargetted(monster, mCastedSpellSchool))
				{
					//TODO: TurtleBrains: Should this automatically remove entity from manager on delete ... probably!
					//Also note, if entity was added by reference overload, then attempted to be removed and then deleted,
					//there will be a crash because the delay from the RemoveEntity() call and the SafeToRemoveEntities()
					//actually performing the removal...
					//monster.GetEntityManager()->RemoveEntity(&monster); //this happens from monster...

					monster.SetToDie();
					itr = mCombatTargets.erase(itr);
					GameResults::IncrementMonsterKillCount();
				}
				else
				{
					++itr;
				}
			}
		}
		else /* Finished dealing damage. */
		{
			//No more target...
			ClearTargettedMonster();
			mCastedSpellSchool = SpellSchool::kInvalid;
		}
	}

	if (false == mKillPlayerTimer.IsZero())
	{
		if (false == mKillPlayerTimer.DecrementStep())
		{
			for (auto itr = mCombatTargets.begin(); itr != mCombatTargets.end(); /* in loop */)
			{
				(*itr)->SetToAttack();
				itr = mCombatTargets.erase(itr);
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnUpdate(const float deltaTime)
{
	if (false == mCanTargetMonster)
	{
		ShowNoAvailableTargets();
		ClearTargettedMonster();
	}
	else
	{
		if (true == tbApplication::Input::IsKeyPressed(tbApplication::tbMouseLeft))
		{
			for (MonsterEntity* monster : mCombatTargets)
			{
				if (monster->IsMouseOver())
				{
					SetTargetToMonster(*monster);
				}
			}
		}

		ShowAllAvailableTargets();
	}

	tbGame::Entity::OnUpdate(deltaTime);
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnRender(void) const
{
	tbGame::Entity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void HauntedRoomEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	tbGame::Entity::OnCollideWith(otherEntity);
}

//--------------------------------------------------------------------------------------------------------------------//
