///
/// @file
/// @details An entity within the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "button_entity.h"

//--------------------------------------------------------------------------------------------------------------------//

ButtonEntity::ButtonEntity(void) :
	tbGame::Entity("ButtonEntity"),

	mButtonSprite(tbGraphics::theSpriteManager.GetSprite("concept_art", "solid_box")),
	mDisabledSprite(tbGraphics::theSpriteManager.GetSprite("concept_art", "outline_box")),
	mHoverOverSprite(tbGraphics::theSpriteManager.GetSprite("concept_art", "solid_pickup")),
	mOnMoveOverAudioEvent("button_over"),
	mOnMoveOutAudioEvent("button_out"),
	mOnClickAudioEvent("button_click"),

	mCurrentState(ButtonState::kDisabled), //Purposely start disabled, to enable in SetState() within ctor call.
	mInputOffset(),
	mButtonTopLeft(),
	mButtonBottomRight()
{
	//tbGraphics::BoxShape* boxShape = new tbGraphics::BoxShape(100.0f, 100.0f, tbGraphics::Color::kRed);
	//boxShape->SetOrigin(tbGraphics::kAnchorCenter);
	//AddGraphic(boxShape);

	AddGraphic(mButtonSprite);
	AddGraphic(mDisabledSprite);
	AddGraphic(mHoverOverSprite);

	SetOrigin(tbGraphics::kAnchorTopLeft);
	SetButtonState(ButtonState::kEnabled);
}

//--------------------------------------------------------------------------------------------------------------------//

ButtonEntity::~ButtonEntity(void)
{
	RemoveGraphic(&mButtonSprite);
	RemoveGraphic(&mDisabledSprite);
	RemoveGraphic(&mHoverOverSprite);
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnAdded(void)
{
	tbGame::Entity::OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnRemoved(void)
{
	tbGame::Entity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnSimulate(void)
{
	tbGame::Entity::OnSimulate();
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnUpdate(const float deltaTime)
{
	tbGame::Entity::OnUpdate(deltaTime);

	if (ButtonState::kDisabled != mCurrentState)
	{
		if (true == IsMouseOver())
		{
			SetButtonState(ButtonState::kHoverOver);
			if (tbApplication::Input::IsKeyReleased(tbApplication::tbMouseLeft))
			{
				tbAudio::theAudioManager.PlayEvent("audio_events", mOnClickAudioEvent);
				OnClicked();
			}
		}
		else
		{
			SetButtonState(ButtonState::kEnabled);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnRender(void) const
{
	tbGame::Entity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	tbGame::Entity::OnCollideWith(otherEntity);
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::SetButtonSprites(const tbGraphics::Sprite& buttonSprite, const tbGraphics::Sprite& hoverSprite, const tbGraphics::Sprite& disabledSprite)
{
	mButtonSprite = buttonSprite;
	mHoverOverSprite = hoverSprite;
	mDisabledSprite = disabledSprite;
	SetOrigin(tbGraphics::kAnchorCenter);
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::SetOrigin(const tbGraphics::AnchorLocation& anchor)
{
	mButtonSprite.SetOrigin(anchor);
	mDisabledSprite.SetOrigin(anchor);
	mHoverOverSprite.SetOrigin(anchor);

	const float buttonWidth(mButtonSprite.GetWidth());
	const float buttonHeight(mButtonSprite.GetHeight());

	switch (anchor)
	{
	default:
	case tbGraphics::kAnchorCenter:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth / 2.0f, -buttonHeight / 2.0f);
		mButtonBottomRight = tbMath::Vector2(buttonWidth / 2.0f, buttonHeight / 2.0f);
		break;
	case tbGraphics::kAnchorTopLeft:
		mButtonTopLeft = tbMath::Vector2(0.0f, 0.0f);
		mButtonBottomRight = tbMath::Vector2(buttonWidth, buttonHeight);
		break;
	case tbGraphics::kAnchorTopCenter:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth / 2.0f, 0.0f);
		mButtonBottomRight = tbMath::Vector2(buttonWidth / 2.0f, buttonHeight);
		break;
	case tbGraphics::kAnchorTopRight:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth, 0.0f);
		mButtonBottomRight = tbMath::Vector2(0.0f, buttonHeight);
		break;
	case tbGraphics::kAnchorCenterLeft:
		mButtonTopLeft = tbMath::Vector2(0.0f, -buttonHeight / 2.0f);
		mButtonBottomRight = tbMath::Vector2(buttonWidth, buttonHeight / 2.0f);
		break;
	case tbGraphics::kAnchorCenterRight:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth, -buttonHeight / 2.0f);
		mButtonBottomRight = tbMath::Vector2(0.0f, buttonHeight / 2.0f);
		break;
	case tbGraphics::kAnchorBottomLeft:
		mButtonTopLeft = tbMath::Vector2(0.0f, -buttonHeight);
		mButtonBottomRight = tbMath::Vector2(buttonWidth, 0.0f);
		break;
	case tbGraphics::kAnchorBottomCenter:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth / 2.0f, -buttonHeight);
		mButtonBottomRight = tbMath::Vector2(buttonWidth / 2.0f, 0.0f);
		break;
	case tbGraphics::kAnchorBottomRight:
		mButtonTopLeft = tbMath::Vector2(-buttonWidth, -buttonHeight);
		mButtonBottomRight = tbMath::Vector2(0.0f, 0.0f);
		break;
	};
}

//--------------------------------------------------------------------------------------------------------------------//

bool ButtonEntity::IsMouseOver(void) const
{
	int mouseX(0);
	int mouseY(0);

	tbApplication::Input::GetMousePosition(&mouseX, &mouseY);
	const tbMath::Vector2 mousePosition(static_cast<float>(mouseX), static_cast<float>(mouseY));
	const tbMath::Vector2& position(GetPosition() + mInputOffset);

	if (mousePosition.x >= position.x + mButtonTopLeft.x && mousePosition.x <= position.x + mButtonBottomRight.x &&
		mousePosition.y >= position.y + mButtonTopLeft.y && mousePosition.y <= position.y + mButtonBottomRight.y)
	{
		return true;
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::SetButtonState(const ButtonState& buttonState)
{
	if (mCurrentState == buttonState)
	{
		return;
	}

	const ButtonState previousState(mCurrentState);
	mCurrentState = buttonState;

	mButtonSprite.SetVisible(false);
	mDisabledSprite.SetVisible(false);
	mHoverOverSprite.SetVisible(false);

	if (ButtonState::kDisabled == buttonState)
	{
		mDisabledSprite.SetVisible(true);
	}
	else if (ButtonState::kHoverOver == buttonState)
	{
		mHoverOverSprite.SetVisible(true);
		tbAudio::theAudioManager.PlayEvent("audio_events", mOnMoveOverAudioEvent);
	}
	else
	{
		tb_error_if(buttonState != ButtonState::kEnabled, "Expected.");
		mButtonSprite.SetVisible(true);

		if (ButtonState::kDisabled != previousState)
		{
			tbAudio::theAudioManager.PlayEvent("audio_events", mOnMoveOutAudioEvent);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

const ButtonEntity::ButtonState& ButtonEntity::GetButtonState(void) const
{
	return mCurrentState;
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::SetInputOffset(const tbMath::Vector2& inputOffset)
{
	mInputOffset = inputOffset;
}

//--------------------------------------------------------------------------------------------------------------------//

void ButtonEntity::OnClicked(void)
{
	//Should be overridden to do interesting things.
}

//--------------------------------------------------------------------------------------------------------------------//
