///
/// @file
/// @details This scene is the gameplay scene for the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "gameplay_scene.h"
#include "scene_manager.h"
#include "score_display.h"
#include "game_results.h"

#include "button_entity.h"
#include "spellbook_entity.h"

//--------------------------------------------------------------------------------------------------------------------//

GameplayScene::GameplayScene(void) :
	tbGame::GameScene(),
	mStayOrLeavePrompt(*this),
	mSpellbook(),
	mHauntedRoom()
{
	AddEntity(mStayOrLeavePrompt);
	AddEntity(mHauntedRoom);
	AddEntity(mSpellbook);
}

//--------------------------------------------------------------------------------------------------------------------//

GameplayScene::~GameplayScene(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::StayInRoomForAnotherHour(void)
{
	tb_log("Spending another hour in the room, you are going mad.\n");
	GameResults::IncrementHoursSpentInRoom();
	const uint32_t numberHoursSpentInRoom(GameResults::GetHoursSpentInRoom());
	if (1 == numberHoursSpentInRoom)
	{	//Only regenerate the spells the first combat after entering the room.
		mSpellbook.RegenerateSpellLists();
	}

	mStayOrLeavePrompt.HidePrompt();
	mHauntedRoom.EnterCombat(numberHoursSpentInRoom);
	mSpellbook.SetCanPrepareSpell(false);    //Also done within regenerate spell list...
	mHauntedRoom.SetCanTargetMonster(false); //Also done within EnterCombat...
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::LeaveRoomLikeAChicken(void)
{
	//TODO: TIM: Goto results screen...

	GameResults::SetWinFlag(true);
	LudumDare37::theSceneManager->ChangeToScene(kResultsScene);
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::OnSimulate(void)
{
	tbGame::GameScene::OnSimulate();
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::OnUpdate(const float deltaTime)
{
	tbGame::GameScene::OnUpdate(deltaTime);

	if (true == tbApplication::Input::IsKeyReleased(tbApplication::tbKeyEscape))
	{	//This should be the very last check to happen as game ends immediately.
		LudumDare37::theSceneManager->ChangeToScene(kTitleScene);
	}

	if (false == mHauntedRoom.IsInCombat() && false == mHauntedRoom.IsDamageBeingDealt() && false == mSpellbook.IsSpellBeingCasted() && false == mHauntedRoom.IsKillingStudent() && true == mHauntedRoom.CanKillStudent())
	{
		if (false == mStayOrLeavePrompt.IsPromptShowing())
		{
			mStayOrLeavePrompt.ShowPrompt();
			mStayOrLeavePrompt.SetHoursInRoom(GameResults::GetHoursSpentInRoom());

			mHauntedRoom.RecalculateNextCombatMonsterCount();
			mStayOrLeavePrompt.SetEnemiesInRoom(mHauntedRoom.NumberMonstersInRoomForNextCombat());

			if (0 == GameResults::GetHoursSpentInRoom())
			{	//Automatically throw the player into the room for another hour here.
				StayInRoomForAnotherHour();
			}
		}

		//This eventually calls "StayInRoomForAnotherHour()" which will then begin combat and continue.
	}
	else if (true == mHauntedRoom.IsInCombat() || false == mHauntedRoom.CanKillStudent())
	{
		if (true /* == some intro sequence has completed... */)
		{
			mSpellbook.SetCanPrepareSpell(true);
			mHauntedRoom.SetCanTargetMonster(false);
		}

		if (false == mSpellbook.IsSpellPrepped())
		{
			//Wait until a spell is prepared.
			if (mSpellbook.NumberOfSpellsRemaining() <= 0)
			{
				//TODO: Need some form of death feedback...

				if (false == mSpellbook.IsSpellBeingCasted() && false == mHauntedRoom.IsDamageBeingDealt())
				{
					mHauntedRoom.MonstersAttackStudent();
					if (false == mHauntedRoom.IsKillingStudent())
					{
						GameResults::SetWinFlag(false);
						LudumDare37::theSceneManager->ChangeToScene(kResultsScene);
					}
				}
			}
		}
		else if (true == mSpellbook.IsSpellPrepped() && true /* == any potential preparation spell sequence */)
		{
			mHauntedRoom.SetCanTargetMonster(true);
			mHauntedRoom.SetPreppedSpellSchool(mSpellbook.GetPreppedSpellSchool());
		}

		if (true == mHauntedRoom.IsInCombat())
		{
			if (false == mHauntedRoom.IsMonsterTargetted())
			{
				//Wait until a monster is targeted.
				tb_error_if(mHauntedRoom.NumberOfMonsters() <= 0, "This shouldn't happen!");
			}
			else if (true == mHauntedRoom.IsMonsterTargetted() && true /* == any potential targeting sequence */)
			{
				//TODO: LudumDare37: USE Spell or something...

				mHauntedRoom.SetCastedSpellSchool(mSpellbook.GetPreppedSpellSchool());
				mSpellbook.CastPreppedSpell();
				mHauntedRoom.SpellCastedOnTargets();
			}
		}
	}

	//TODO: TemplateProject should include this automatically.
	tbGraphics::GifPerformCapture(deltaTime, tbApplication::Input::IsKeyPressed(tbApplication::tbKeyG));
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::OnRender(void) const
{
	tbGame::GameScene::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::OnOpen(void)
{
	tbGame::GameScene::OnOpen();

	GameResults::ResetGameResults(false); //Must be before hauntedroom resets.
	mHauntedRoom.ResetToEmptyRoom();
	mSpellbook.ResetToEmptyBook();
	mStayOrLeavePrompt.ResetToInitial();	
}

//--------------------------------------------------------------------------------------------------------------------//

void GameplayScene::OnClose(void)
{
	tbGame::GameScene::OnClose();
}

//--------------------------------------------------------------------------------------------------------------------//
