///
/// @file
/// @details An entity within the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "monster_entity.h"

//--------------------------------------------------------------------------------------------------------------------//

class MonsterPlayAnimationBehavior : public tbGame::EntityBehaviorInterface
{
public:
	MonsterPlayAnimationBehavior(MonsterEntity& monster, const tbCore::tbString& animationName, const bool isLooping) :
		tbGame::EntityBehaviorInterface(monster),
		mMonsterEntity(monster),
		mAnimationName(animationName),
		mIsLooping(isLooping)
	{
	}

	virtual ~MonsterPlayAnimationBehavior(void)
	{
	}

	virtual void OnSimulate(void) override
	{
		mMonsterEntity.PlayAnimation(mAnimationName, mIsLooping);
		mMonsterEntity.PopBehavior();
	}

private:
	MonsterEntity& mMonsterEntity;
	const tbCore::tbString mAnimationName;
	const bool mIsLooping;
};

//--------------------------------------------------------------------------------------------------------------------//

class MonsterGetBigBehavior : public tbGame::EntityBehaviorInterface
{
public:
	MonsterGetBigBehavior(MonsterEntity& monster, tbGraphics::Sprite& sprite) :
		tbGame::EntityBehaviorInterface(monster),
		mMonsterEntity(monster),
		mMonsterSprite(sprite),
		mInitialScale(0.75f),
		mFinalScale(1.25f),
		mTimer(2500)
	{
	}

	virtual ~MonsterGetBigBehavior(void)
	{
	}

	virtual void OnSimulate(void) override
	{
		mMonsterEntity.PlayAnimation("targeted");

		const float percentage = 1.0f - mTimer.GetPercentageOf(tbGame::GameTimer(2500));
		mMonsterSprite.SetScale(tbMath::Interpolation::Linear(percentage, mInitialScale, mFinalScale));

		if (true == mTimer.DecrementStep())
		{
			mMonsterEntity.PopBehavior();
		}
	}

private:
	MonsterEntity& mMonsterEntity;
	tbGraphics::Sprite& mMonsterSprite;
	const float mInitialScale;
	const float mFinalScale;
	tbGame::GameTimer mTimer;
};

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString MonsterTypeToString(const MonsterType& monsterType)
{
	switch (monsterType)
	{
	case kMonsterSlimeBall: return tb_string("slime_ball");
	case kMonsterGoblin: return tb_string("goblin");
	case kMonsterSkeleton: return tb_string("skeleton");
	case kMonsterTreasureChest: return tb_string("treasure_chest");
	};

	tb_error("Unexpected monsterType!");
	return tb_string("");
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString AnimationPrefixFromMonsterType(const MonsterType& monsterType)
{
	return MonsterTypeToString(monsterType) + tb_string("_");
}

//--------------------------------------------------------------------------------------------------------------------//

MonsterEntity::MonsterEntity(HauntedRoomEntity& hauntedRoom, const MonsterType& monsterType) :
	TargetEntity(hauntedRoom),
	mMonsterType(monsterType),
	mAnimationPrefix(AnimationPrefixFromMonsterType(monsterType)),
	mMonsterSprite("monster_sheet", MonsterTypeToString(monsterType))
{
	PlayAnimation("idle", true);
	mMonsterSprite.SetScale(0.75f);
	AddGraphic(mMonsterSprite);
}

//--------------------------------------------------------------------------------------------------------------------//

MonsterEntity::~MonsterEntity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void MonsterEntity::SetToAttack(void)
{
	//Must be exactly 4s

	const int randomTimer(tbMath::RandomInt(50, 150) * 10);

	PushBehavior(new tbGame::KillBehavior(*this));
	PushBehavior(new tbGame::DelayBehavior(*this, tbGame::GameTimer(1500 - randomTimer)));
	PushBehavior(new MonsterGetBigBehavior(*this, mMonsterSprite)); //2500ms
	PushBehavior(new tbGame::DelayBehavior(*this, tbGame::GameTimer(randomTimer)));
}

//--------------------------------------------------------------------------------------------------------------------//

void MonsterEntity::SetToDie(void)
{
	PlayAnimation("targeted", true);

	//Take at most 3s
	PushBehavior(new tbGame::KillBehavior(*this));
	PushBehavior(new tbGame::DelayBehavior(*this, tbGame::GameTimer(tbMath::RandomInt(100, 150) * 10)));
	PushBehavior(new MonsterPlayAnimationBehavior(*this, tb_string("death"), true));
	PushBehavior(new tbGame::DelayBehavior(*this, tbGame::GameTimer(tbMath::RandomInt(50, 150) * 10)));
}

//--------------------------------------------------------------------------------------------------------------------//

void MonsterEntity::PlayAnimation(const tbCore::tbString& animationName, const bool isLooping, const bool isForward, const float timePerFrame)
{
	mMonsterSprite.PlayAnimation(mAnimationPrefix + animationName, isLooping, isForward, timePerFrame);
}

//--------------------------------------------------------------------------------------------------------------------//
