///
/// @file
/// @details Provide a simple title scene for the LudumDare37 project that will fade out into gameplay.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "results_scene.h"
#include "scene_manager.h"
#include "game_results.h"

const tbGame::GameTimer kMaximumFadeInTime(500);
const tbGame::GameTimer kMaximumFadeOutTime(500);

//--------------------------------------------------------------------------------------------------------------------//

ResultsScene::ResultsScene(void) :
	tbGame::GameScene(),
	mWinBackdrop(tbGraphics::Color(0xFF008000)),
	mLoseBackdrop(tbGraphics::Color(0xFF800000)),
	mResultsText("You Survived", 75.0f),
	mFadeInTimer(kMaximumFadeInTime),
	mFadeOutTimer(0),
	mBackToTitleAction()
{
	mBackToTitleAction.AddBinding(tbApplication::tbKeyEscape);
	mBackToTitleAction.AddBinding(tbApplication::tbKeySpace);
	mBackToTitleAction.AddBinding(tbApplication::tbMouseLeft);

	mWinBackdrop.SetDepth(-0.5f);
	AddGraphic(mWinBackdrop);

	mLoseBackdrop.SetDepth(-0.5f);
	AddGraphic(mLoseBackdrop);

	mResultsText.SetOrigin(tbGraphics::kAnchorTopCenter);
	mResultsText.SetPosition(tbGraphics::ScreenCenter().x, 25.0f);
	AddGraphic(mResultsText);
}

//--------------------------------------------------------------------------------------------------------------------//

ResultsScene::~ResultsScene(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void ResultsScene::OnSimulate(void)
{
	if (false == mFadeInTimer.IsZero())
	{
		float percentage = mFadeInTimer.GetPercentageOf(kMaximumFadeInTime);
		tbGraphics::Color fadingColor;
		fadingColor.SetColor(percentage, tbGraphics::Color::kAlphaFull, tbGraphics::Color::kAlphaNone);
		SetColor(fadingColor);

		if (true == mFadeInTimer.DecrementStep())
		{
			SetColor(tbGraphics::Color::kAlphaFull);
		}
	}

	if (false == mFadeOutTimer.IsZero())
	{
		float percentage = mFadeOutTimer.GetPercentageOf(kMaximumFadeOutTime);
		tbGraphics::Color fadingColor;
		fadingColor.SetColor(percentage, tbGraphics::Color::kAlphaNone, tbGraphics::Color::kAlphaFull);
		SetColor(fadingColor);

		if (true == mFadeOutTimer.DecrementStep())
		{
			SetColor(tbGraphics::Color::kAlphaNone);
			LudumDare37::theSceneManager->ChangeToScene(SceneId::kTitleScene);
		}
	}
	else
	{
		if (true == mBackToTitleAction.IsPressed())
		{
			mFadeOutTimer = kMaximumFadeOutTime;
		}
	}

	tbGame::GameScene::OnSimulate();
}

//--------------------------------------------------------------------------------------------------------------------//

void ResultsScene::OnUpdate(const float deltaTime)
{
	tbGame::GameScene::OnUpdate(deltaTime);

	const bool didWin = GameResults::GetWinFlag();
	mWinBackdrop.SetVisible(didWin);
	mWinBackdrop.SetColor(tbGraphics::Color(0xFF008000));
	mLoseBackdrop.SetVisible(!didWin);
	mLoseBackdrop.SetColor(tbGraphics::Color(0xFF800000));

	mResultsText.SetText((didWin) ? tb_string("You Survived") : tb_string("You Collapsed"));
	mResultsText.SetOrigin(tbGraphics::kAnchorTopCenter);
}

//--------------------------------------------------------------------------------------------------------------------//

void ResultsScene::OnRender(void) const
{
	tbGame::GameScene::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void ResultsScene::OnOpen(void)
{
	mFadeInTimer = kMaximumFadeInTime;
	mFadeOutTimer = 0;
	SetColor(tbGraphics::Color::kAlphaNone);

	ScoringResult* hoursSpent = new ScoringResult;
	hoursSpent->mDescriptionText.SetText("Hours Spent in Room:");
	hoursSpent->mScoreValue = GameResults::GetHoursSpentInRoom();
	hoursSpent->mBestValue = GameResults::GetMostHoursSpentInRoom();
	hoursSpent->mCurrentValue = hoursSpent->mScoreValue;
	mResultingScores.push_back(hoursSpent);

	ScoringResult* monstersKilled = new ScoringResult;
	monstersKilled->mDescriptionText.SetText("Monsters Killed:");
	monstersKilled->mScoreValue = GameResults::GetMonsterKillCount();
	monstersKilled->mBestValue = GameResults::GetBestMonsterKillCount();
	monstersKilled->mCurrentValue = monstersKilled->mScoreValue;
	mResultingScores.push_back(monstersKilled);

	float offsetX = 325.0f;
	float currentY = 180.0f;
	const float changeY = 25.0f;
	for (ScoringResult* result : mResultingScores)
	{
		result->mScoreText.SetText(tb_string(result->mCurrentValue));
		result->mBestScoreText.SetText("(" + tb_string(result->mBestValue) + ")");

		result->mDescriptionText.SetPosition(100.0f + offsetX, currentY);
		result->mScoreText.SetPosition(400.0f + offsetX, currentY);
		result->mBestScoreText.SetPosition(500.0f + offsetX, currentY);
		currentY += changeY;

		AddGraphic(result->mDescriptionText);
		AddGraphic(result->mScoreText);
		AddGraphic(result->mBestScoreText);
	}

	const bool didWin = GameResults::GetWinFlag();
	tb_log("Game Results didWin: %d\n", (didWin) ? 1 : 0);
	mWinBackdrop.SetVisible(didWin);
	mLoseBackdrop.SetVisible(!didWin);
}

//--------------------------------------------------------------------------------------------------------------------//

void ResultsScene::OnClose(void)
{
	for (ScoringResult* result : mResultingScores)
	{
		RemoveGraphic(&result->mDescriptionText);
		RemoveGraphic(&result->mScoreText);
		RemoveGraphic(&result->mBestScoreText);

		delete result;
	}

	mResultingScores.clear();

	tbGame::GameScene::OnClose();
}

//--------------------------------------------------------------------------------------------------------------------//

