///
/// @file
/// @details An entity within the LudumDare37 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "spellbook_entity.h"

const tbGame::GameTimer kMaximumSpellCastingTime(1000);

//--------------------------------------------------------------------------------------------------------------------//

SpellbookEntity::SpellbookEntity(void) :
	tbGame::Entity("SpellbookEntity"),
	mSpellbookSprite(tbGraphics::theSpriteManager.GetSprite("spellbook_sheet", "spellbook")),
	mSpellSchoolButtons{
		new SpellbookButtonEntity(*this, SpellSchool::kFire),
		new SpellbookButtonEntity(*this, SpellSchool::kWater),
		new SpellbookButtonEntity(*this, SpellSchool::kEarth),
		new SpellbookButtonEntity(*this, SpellSchool::kMind),
	},
	mCanPrepareSpell(false),
	mIsSpellPrepped(false),
	mSpellCastingTimer(tbGame::GameTimer::Zero()),
	mPreppedSpellSchool(SpellSchool::kInvalid)
{
	SetPosition(tbGraphics::ScreenCenter().x, 50.0f);

	mSpellbookSprite.SetOrigin(tbGraphics::kAnchorTopCenter);
	AddGraphic(mSpellbookSprite);
}

//--------------------------------------------------------------------------------------------------------------------//

SpellbookEntity::~SpellbookEntity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::ResetToEmptyBook(void)
{
	ClearPreppedSpell(false);

	for (size_t schoolIndex(0); schoolIndex < kMaximumSpellSchools; ++schoolIndex)
	{
		mSpellSchoolButtons[schoolIndex]->SetAsPreppedSpell(false);
		mSpellSchoolButtons[schoolIndex]->SetUsesAvailable(0);
	}

	mCanPrepareSpell = false;
	mIsSpellPrepped = false;
	mSpellCastingTimer = tbGame::GameTimer::Zero();
	mPreppedSpellSchool = SpellSchool::kInvalid;
}

//--------------------------------------------------------------------------------------------------------------------//

bool SpellbookEntity::IsSpellPrepped(void) const
{
	return mIsSpellPrepped;
}

//--------------------------------------------------------------------------------------------------------------------//

bool SpellbookEntity::IsSpellBeingCasted(void) const
{
	return mSpellCastingTimer.IsZero() ? false : true;
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::CastPreppedSpell(void)
{
	mSpellCastingTimer = kMaximumSpellCastingTime;
	ClearPreppedSpell(true);
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::ClearPreppedSpell(const bool byUsing)
{
	if (true == IsSpellPrepped())
	{
		if (true == byUsing)
		{
			const size_t usedSchoolIndex(SpellSchoolToIndex(mPreppedSpellSchool));
			int schoolUsesAvailable = mSpellSchoolButtons[usedSchoolIndex]->GetUsesAvailable() - 1;
			tb_error_if(schoolUsesAvailable < 0, "This should never happen...");

			mSpellSchoolButtons[usedSchoolIndex]->SetUsesAvailable(schoolUsesAvailable);

			tb_log("Using spell from %s school, spells remaining:\n", SpellSchoolToString(mPreppedSpellSchool).c_str());
			for (size_t index(0); index < kMaximumSpellSchools; ++index)
			{
				tb_log("    %s school: %d\n", SpellSchoolToString(IndexToSpellSchool(index)).c_str(), mSpellSchoolButtons[index]->GetUsesAvailable());
			}
		}

		for (SpellbookButtonEntity* spellButton : mSpellSchoolButtons)
		{
			spellButton->SetAsPreppedSpell(false);
		}

		mIsSpellPrepped = false;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::PrepareSpellFromSchool(const SpellSchool& spellSchool)
{
	//tb_error_if(IsSpellPrepped(), "Calling PrepareSpell when a spell is already prepped, clear it first.");
	ClearPreppedSpell(false);

	tb_log("Preparing a spell from: %s", SpellSchoolToString(spellSchool).c_str());
	mIsSpellPrepped = true;
	mPreppedSpellSchool = spellSchool;

	mSpellSchoolButtons[SpellSchoolToIndex(mPreppedSpellSchool)]->SetAsPreppedSpell(true);
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::SetCanPrepareSpell(const bool canPrepareSpell)
{
	if (mCanPrepareSpell != canPrepareSpell)
	{
		for (SpellbookButtonEntity* spellButton : mSpellSchoolButtons)
		{
			spellButton->SetCanBePrepped(canPrepareSpell);
		}
	}

	mCanPrepareSpell = canPrepareSpell;
}

//--------------------------------------------------------------------------------------------------------------------//

int SpellbookEntity::NumberOfSpellsRemaining(void) const
{
	int totalSpellsRemaining(0);
	for (const SpellbookButtonEntity* spellButton : mSpellSchoolButtons)
	{
		totalSpellsRemaining += spellButton->GetUsesAvailable();
	}
	return totalSpellsRemaining;
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnAdded(void)
{
	tbGame::Entity::OnAdded();

	tb_error_if(kMaximumSpellSchools != mSpellSchoolButtons.size(), "Expected this to match.");
	for (size_t schoolIndex(0); schoolIndex < kMaximumSpellSchools; ++schoolIndex)
	{
		GetEntityManager()->AddEntity(*mSpellSchoolButtons[schoolIndex]);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnRemoved(void)
{
	tb_error_if(kMaximumSpellSchools != mSpellSchoolButtons.size(), "Expected this to match.");
	for (size_t schoolIndex(0); schoolIndex < kMaximumSpellSchools; ++schoolIndex)
	{
		GetEntityManager()->RemoveEntity(mSpellSchoolButtons[schoolIndex]);
	}

	tbGame::Entity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnSimulate(void)
{
	tbGame::Entity::OnSimulate();

	if (false == mSpellCastingTimer.IsZero())
	{
		if (false == mSpellCastingTimer.DecrementStep())
		{	//Do some effects!
			//mSpellbookSprite.SetColor(tbGraphics::Color(0xFFFF0000));
		}
		else /* Finished casting the spell. */
		{
			//Finished.
			//mSpellbookSprite.SetColor(tbGraphics::Color::kWhite);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnUpdate(const float deltaTime)
{
	tbGame::Entity::OnUpdate(deltaTime);
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnRender(void) const
{
	tbGame::Entity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	tbGame::Entity::OnCollideWith(otherEntity);
}

//--------------------------------------------------------------------------------------------------------------------//

void SpellbookEntity::RegenerateSpellLists(void)
{
	const size_t kNumberOfRolls(10);
	for (size_t diceIndex(0); diceIndex < kNumberOfRolls; ++diceIndex)
	{
		const size_t schoolIndex(tbMath::RandomInt() % kMaximumSpellSchools);
		//const size_t schoolIndex(SpellSchoolToIndex(SpellSchool::kMind)); //For maximized testing

		int usesAvailable = mSpellSchoolButtons[schoolIndex]->GetUsesAvailable() + 1;
		mSpellSchoolButtons[schoolIndex]->SetUsesAvailable(usesAvailable);
		
	}

	SetCanPrepareSpell(false);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

size_t SpellSchoolToIndex(const SpellSchool& spellSchool)
{
	return (SpellSchool::kFire == spellSchool) ? 0 :
		(SpellSchool::kWater == spellSchool) ? 1 :
		(SpellSchool::kEarth == spellSchool) ? 2 :
		(SpellSchool::kMind == spellSchool) ? 3 :
		0; //Should never actually reach this.
}

//--------------------------------------------------------------------------------------------------------------------//

SpellSchool IndexToSpellSchool(const size_t& schoolIndex)
{
	return (0 == schoolIndex) ? SpellSchool::kFire :
		(1 == schoolIndex) ? SpellSchool::kWater :
		(2 == schoolIndex) ? SpellSchool::kEarth :
		(3 == schoolIndex) ? SpellSchool::kMind :
		SpellSchool::kInvalid; //Should never actually reach this.
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString SpellSchoolToString(const SpellSchool& spellSchool)
{
	return (SpellSchool::kFire == spellSchool) ? tb_string("Fire") :
		(SpellSchool::kWater == spellSchool) ? tb_string("Water") :
		(SpellSchool::kEarth == spellSchool) ? tb_string("Earth") :
		(SpellSchool::kMind == spellSchool) ? tb_string("Mind") :
		tb_string("Unknown"); //Should never actually reach this.
}

//--------------------------------------------------------------------------------------------------------------------//
